/*
 * Panasonic XHCI Host Controller Driver
 *
 * Copyright (C) 2012-2020 Panasonic Corporation.
 *
 * This file is licensed under  the terms of the GNU General Public
 * License version 2. This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#include <linux/module.h>
#include <linux/slab.h>
#include "xhci.h"
#include "xhci-panasonic.h"
#include <linux/platform_device.h>
#include "../core/usb-panasonic.h"

/* Linux3　対応 */
#define DMA_BIT_MASK(n)	(((n) == 64) ? ~0ULL : ((1ULL<<(n))-1))


struct mutex xhci_wr_mutex;

static const char hcd_name[] = "xhci_hcd";
///struct timeval polling_start[5] = { {NULL,NULL},{NULL,NULL},{NULL,NULL},{NULL,NULL},{NULL,NULL} };


static void xhci_pci_quirks(struct device *dev, struct xhci_hcd *xhci)
{
	return;
}

/*_---------------------------------------------------------------------------
     Panasonic xHCI用 初期化処理
---------------------------------------------------------------------------_*/
/* called during probe() after chip reset completes */
static int xhci_panasonic_setup(struct usb_hcd *hcd)
{
	struct xhci_hcd		*xhci;
	int		retval;

	retval = xhci_gen_setup(hcd, xhci_pci_quirks);
	if (retval)
		return retval;

	xhci = hcd_to_xhci(hcd);

	return 0;

#if 0
/* Commit:b02d0ed677acb3465e7600366f2353413bf24074		xhci: Change hcd_priv into a pointer */
	if (usb_hcd_is_primary_hcd(hcd)) {
		xhci = kzalloc(sizeof(struct xhci_hcd), GFP_KERNEL);
		if (!xhci)
			return -ENOMEM;
		*((struct xhci_hcd **) hcd->hcd_priv) = xhci;
		xhci->main_hcd = hcd;
		/* Mark the first roothub as being USB 2.0.
		 * The xHCI driver will register the USB 3.0 roothub.
		 */
		hcd->speed = HCD_USB2;
		hcd->self.root_hub->speed = USB_SPEED_HIGH;
		/*
		 * USB 2.0 roothub under xHCI has an integrated TT,
		 * (rate matching hub) as opposed to having an OHCI/UHCI
		 * companion controller.
		 */
		hcd->has_tt = 1;
	} else {
		/* xHCI private pointer was set in xhci_pci_probe for the second
		 * registered roothub.
		 */
		xhci = hcd_to_xhci(hcd);

/* ★★対応いるか要検討 @20120229 for usb debug */
#if 0
		temp = xhci_readl(xhci, &xhci->cap_regs->hcc_params);
		if (HCC_64BIT_ADDR(temp)) {
			xhci_dbg(xhci, "Enabling 64-bit DMA addresses.\n");
			dma_set_mask(hcd->self.controller, DMA_BIT_MASK(64));
		} else {
			dma_set_mask(hcd->self.controller, DMA_BIT_MASK(32));
		}
#endif
		return 0;
	}


	xhci->cap_regs = hcd->regs;
	xhci->op_regs = hcd->regs +
		HC_LENGTH(xhci_readl(xhci, &xhci->cap_regs->hc_capbase));
	xhci->run_regs = hcd->regs +
		(xhci_readl(xhci, &xhci->cap_regs->run_regs_off) & RTSOFF_MASK);
	/* Cache read-only capability registers */
	xhci->hcs_params1 = xhci_readl(xhci, &xhci->cap_regs->hcs_params1);
	xhci->hcs_params2 = xhci_readl(xhci, &xhci->cap_regs->hcs_params2);
	xhci->hcs_params3 = xhci_readl(xhci, &xhci->cap_regs->hcs_params3);
	xhci->hcc_params = xhci_readl(xhci, &xhci->cap_regs->hc_capbase);
	xhci->hci_version = HC_VERSION(xhci->hcc_params);
	xhci->hcc_params = xhci_readl(xhci, &xhci->cap_regs->hcc_params);
	xhci_print_registers(xhci);
	
	/* Make sure the HC is halted. */
	retval = xhci_halt(xhci);
	if (retval)
		goto error;

	xhci_dbg(xhci, "Resetting HCD\n");
	/* Reset the internal HC memory state and registers. */
	retval = xhci_reset(xhci);
	if (retval)
		goto error;
	xhci_dbg(xhci, "Reset complete\n");


/* ★★対応いるか要検討 @20120229 for usb debug */
#if 0
	temp = xhci_readl(xhci, &xhci->cap_regs->hcc_params);
	if (HCC_64BIT_ADDR(temp)) {
		xhci_dbg(xhci, "Enabling 64-bit DMA addresses.\n");
		dma_set_mask(hcd->self.controller, DMA_BIT_MASK(64));
	} else {
		dma_set_mask(hcd->self.controller, DMA_BIT_MASK(32));
	}
#endif

	xhci_dbg(xhci, "Calling HCD init\n");
	/* Initialize HCD and host controller data structures. */
	retval = xhci_init(hcd);
	if (!retval)
		return retval;


/* Commit:b02d0ed677acb3465e7600366f2353413bf24074		xhci: Change hcd_priv into a pointer */
error:
	kfree(xhci);
	return retval;

#endif


}


/*_---------------------------------------------------------------------------
     Panasonic hc_driver
---------------------------------------------------------------------------_*/
static const struct hc_driver xhci_panasonic_hc_driver = {
	.description =		hcd_name,
	.product_desc =		"panasonic xHCI Host Controller",
	.hcd_priv_size =	sizeof(struct xhci_hcd *),

	/*
	 * generic hardware linkage
	 */
	.irq =			xhci_irq,
	.flags =		HCD_MEMORY | HCD_USB3,

	/*
	 * basic lifecycle operations
	 */
	.reset =		xhci_panasonic_setup,
	.start =		xhci_run,
	/* suspend and resume implemented later */
	.stop =			xhci_stop,
	.shutdown =		xhci_shutdown,

	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue 	=	xhci_urb_enqueue,
	.urb_dequeue 	=	xhci_urb_dequeue,
	.alloc_dev 		=	xhci_alloc_dev,
	.free_dev 		=	xhci_free_dev,


/*_ Commit:eab1cafc3b524b714b0567ab98fc75ace09db98c			USB: Support for allocating USB 3.0 streams.			2010/04/06 02:55 _*/
	.alloc_streams =	xhci_alloc_streams,
	.free_streams =		xhci_free_streams,


	.add_endpoint 	=	xhci_add_endpoint,
	.drop_endpoint 	=	xhci_drop_endpoint,
	.endpoint_reset =	xhci_endpoint_reset,
	.check_bandwidth =	xhci_check_bandwidth,
	.reset_bandwidth =	xhci_reset_bandwidth,
	.address_device =	xhci_address_device,
	.update_hub_device =	xhci_update_hub_device,

/*_ Commit:a5f0efaba4c2b644e6248648f75b0a8a522359f6			USB: Add call to notify xHC of a device reset.				2009/12/10 08:59 _*/
/*  Commit:f0615c45ce5feb141c1172480c5198d4b8d25436		USB: xHCI: change xhci_reset_device() to allocate new device */
	.reset_device =		xhci_discover_or_reset_device,

	/*
	 * scheduling support
	 */
	.get_frame_number =	xhci_get_frame,

	/* Root hub support */
	.hub_control =		xhci_hub_control,
	.hub_status_data =	xhci_hub_status_data,

	/*
	 * call back when device connected and addressed
	 */
	.update_device 		=   xhci_update_device,
	.set_usb2_hw_lpm 	=	xhci_set_usb2_hardware_lpm,

};

/*_---------------------------------------------------------------------------
     Panasonic xHCI用 Probe処理
---------------------------------------------------------------------------_*/
static int xhci_panasonic_probe(struct platform_device *pdev)
{
	const struct hc_driver *driver = &xhci_panasonic_hc_driver;
	struct usb_hcd	*hcd;
	struct xhci_hcd *xhci;	
	struct resource *res;
	unsigned long	rsrc_start;	/*_  memory/io resource start _*/
	unsigned long	rsrc_len;	/*_  memory/io resource length _*/
	int		irq;
	int		retval=0;
	
	if( usb_disabled() ) return -ENODEV;
	
	/*_ デバイスからIRQを取得する _*/
	res = platform_get_resource( pdev, IORESOURCE_IRQ, 0 );
	assert( res );	/*_ resourceは、デバイス登録時にStaticで登録するため失敗しない(失敗の場合、ロジックエラーかメモリ不足) _*/
	if( !res ) return -ENODEV;
	irq = res->start;
	
	/*_ デバイスからアドレスを取得する _*/
	res = platform_get_resource( pdev, IORESOURCE_MEM, 0 );
	//assert( res );
	if( !res ) return -ENODEV;
	rsrc_start = res->start;
	rsrc_len =   resource_size(res);
	
	/*_ @LINUXHOST:V0303nd[11/12/12] xHCI dma_mask設定場所を変更 _*/
	pdev->dev.coherent_dma_mask=0xffffffffull;
	pdev->dev.dma_mask = &pdev->dev.coherent_dma_mask;

	/*_ hcd作成 _*/
	hcd = usb_create_hcd( driver, &pdev->dev, dev_name(&pdev->dev) );
	if( !hcd ){
		return -ENOMEM;
	}
	
	/*_ hcdにパラメータセット _*/
	hcd->rsrc_start = rsrc_start;
	hcd->rsrc_len = rsrc_len;
	if (!request_mem_region(hcd->rsrc_start, hcd->rsrc_len,
			driver->description)) {
		dev_dbg(&pdev->dev, "controller already in use\n");
		usb_put_hcd(hcd);
		return -EBUSY;
	}

	/*_ レジスタ設定 _*/
	hcd->regs = (struct ehci_regs*)( rsrc_start );
	
#ifdef CONFIG_USB_PANASONIC_XHCI_IOREMAP
	/*_ デバイス終了時に自動的にunmapされる。 _*/
	hcd->regs = devm_ioremap_nocache( &pdev->dev, (resource_size_t)hcd->regs, rsrc_len );
#endif /* CONFIG_USB_PANASONIC_XHCI_IOREMAP */

	/*_ 割込設定 _*/
	hcd->irq = irq;
	
	/*_ hcdを登録 _*/
	retval = usb_add_hcd( hcd, irq, IRQF_SHARED );
	if( retval != 0){
#ifdef CONFIG_USB_PANASONIC_XHCI_IOREMAP
		iounmap(hcd->regs);
#endif /* CONFIG_USB_PANASONIC_XHCI_IOREMAP */
		release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
		usb_put_hcd(hcd);
		return retval;
	}


	/* USB 2.0 roothub is stored in the PCI device now. */
	xhci = hcd_to_xhci(hcd);
	xhci->shared_hcd = usb_create_shared_hcd(driver, &pdev->dev,
				dev_name(&pdev->dev), hcd);
	if (!xhci->shared_hcd) {
		retval = -ENOMEM;
		goto dealloc_usb2_hcd;
	}

	/* Set the xHCI pointer before xhci_pci_setup() (aka hcd_driver.reset)
	 * is called by usb_add_hcd().
	 */
	*((struct xhci_hcd **) xhci->shared_hcd->hcd_priv) = xhci;

	retval = usb_add_hcd(xhci->shared_hcd, irq,	IRQF_SHARED);

	if (retval)
		goto put_usb3_hcd;
	/* Roothub already marked as USB 3.0 speed */
	return 0;

put_usb3_hcd:
	usb_put_hcd(xhci->shared_hcd);
dealloc_usb2_hcd:
	local_irq_disable();
	usb_hcd_irq(0, hcd);
	local_irq_enable();
	usb_remove_hcd(hcd);
#ifdef CONFIG_USB_PANASONIC_XHCI_IOREMAP
	iounmap(hcd->regs);
#endif /* CONFIG_USB_PANASONIC_XHCI_IOREMAP */
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
	usb_put_hcd(hcd);
	return retval;
}

/*_---------------------------------------------------------------------------
     Panasonic xHCI用 Remove処理
---------------------------------------------------------------------------_*/
static int xhci_panasonic_remove(struct platform_device *pdev)
{
	struct xhci_hcd *xhci;
	struct usb_hcd *hcd = platform_get_drvdata(pdev);

	xhci=hcd_to_xhci(hcd);
	if (xhci->shared_hcd) {
		usb_remove_hcd(xhci->shared_hcd);
		usb_put_hcd(xhci->shared_hcd);
	}

	local_irq_disable();
	usb_hcd_irq(0, hcd);
	local_irq_enable();

	usb_remove_hcd(hcd);

#ifdef CONFIG_USB_PANASONIC_XHCI_IOREMAP
///	iounmap(hcd->regs);							/* unmapが重複するため実行しない */
#endif /* CONFIG_USB_PANASONIC_XHCI_IOREMAP */
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
	usb_put_hcd(hcd);
	kfree(xhci);
	return 0;
}

/*_---------------------------------------------------------------------------
    Panasonic用 USBドライバ構造体
---------------------------------------------------------------------------_*/
MODULE_ALIAS("platform:panasonic-xhci");

static struct platform_driver panasonic_xhci_driver = {
	.probe = xhci_panasonic_probe,
	.remove = xhci_panasonic_remove,
	.driver = {
		.name = "panasonic-xhci",	/*_ platform_device_register_simpleと名前をあわせる _*/
	},
};

#define HOST_NUM	ARRAY_SIZE( host_res )
static struct platform_device *myDevices[HOST_NUM];

/*_---------------------------------------------------------------------------
    Panasonic用 USBデバイス/ドライバ登録処理
---------------------------------------------------------------------------_*/
int xhci_register_panasonic( void )
{
	int i, isSomeone, ret;

	/*_ @LINUXHOST:V0302nx[11/11/11] ホスト有効/無効の動的切替用 _*/
	uint xhci_info;
	
	mutex_init( &xhci_wr_mutex );

	/*_ @LINUXHOST:V0302nx[11/11/11] ドライババージョン情報の出力 _*/
	printk(KERN_INFO "XHCI hd v%04x\n",USBH_DDVERSION);
	

	/*_ @LINUXHOST:V0302nx[11/11/11] ホスト有効/無効の動的切替用 _*/
	xhci_info = usbh_get_host_enableinfo(USBH_ENABLEINFO_XHCI);
#ifdef CONFIG_USB_PANASONIC_HOST_SUPPORT
	/*_ @LINUXHOST:V0325fx[12/06/28] 不要なログは出力しない _*/
#else
	printk("Enable XHCI Info = %08x\n",xhci_info);
#endif

	/*_  periphery hardware init _*/
	xhci_panasonic_peripheryhw_init();

	/*_ 外部変数の初期化 _*/
///	for( i=0 ; i < 6 ; i++ ){
///		polling_start[i].tv_sec =  NULL;
///		polling_start[i].tv_usec = NULL;
///	}

	/*_ register driver _*/
	ret = platform_driver_register(&panasonic_xhci_driver);
	if( ret < 0 ){
		printk( "%s:%d platform_driver_register(&panasonic_xhci_driver) error ret=%d\n", __FUNCTION__, __LINE__, ret );
		return -1;
	}
	
	
	/*_ register device _*/
	isSomeone=0;
	for( i=0 ; i < HOST_NUM ; i++ ){

		/*_ @LINUXHOST:V0302nx[11/11/11] ホスト有効/無効の動的切替用 _*/
		if( xhci_info & (0x000000001 << i) ){
			printk(" Ignore XHCI No.%d \n",i);
			myDevices[i]=NULL;
			continue;
		}

		assert( myDevices[i] == NULL );
		
		/*_ platform_deviceとして、USBホストデバイスを登録する _*/
		myDevices[i] = platform_device_register_simple( "panasonic-xhci", i, (struct resource*)host_res[i], ARRAY_SIZE(host_res[i]) );
		if( IS_ERR(myDevices[i]) ){
			printk( "%s:%d error i=%d\n", __FUNCTION__, __LINE__, i );
			myDevices[i]=NULL;
			continue;
		}
		isSomeone=1;
	}

	/*_ @LINUXHOST:V0320fx[12/06/20] USB3.0のWarmReset完了待ち 完了後はWRC,RCのBitを0にする _*/
	msleep(120);
	
	if( isSomeone ) return 0;
	return -1;
}

/*_---------------------------------------------------------------------------
    Panasonic用 USBデバイス/ドライバ登録解除処理
---------------------------------------------------------------------------_*/
void xhci_unregister_panasonic( void )
{
	int i;

	/*_ unregister device _*/
	for( i=0 ; i < HOST_NUM ; i++ ){
		if( myDevices[i] == NULL ) continue;
		platform_device_unregister( myDevices[i] );
		myDevices[i] = NULL;
	}
	
	/*_ unregister driver _*/
	platform_driver_unregister(&panasonic_xhci_driver);
	
	/*_  periphery hardware end _*/
	xhci_panasonic_peripheryhw_end();
}


