/*
 * PCIe host controller driver for Panasonic pvc03v SoCs
 *
 * Copyright (C) 2016-2020 Panasonic Electronics Co., Ltd.
 *
 * Author: 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of_gpio.h>
#include <linux/pci.h>
#include <linux/platform_device.h>
#include <linux/resource.h>
#include <linux/signal.h>
#include <linux/types.h>

#include "pcie-designware.h"
#include "common_spin_lock.h"

#define to_pvc03v_pcie(x)	container_of(x, struct pvc03v_pcie, pp)

struct pvc03v_pcie {
	void __iomem		*stu_base;
	void __iomem		*phy_base;
	void __iomem		*block_base;
	int			reset_gpio;
	int			pcie_ch;		/*add ch1*/
	struct clk		*clk;
	struct clk		*bus_clk;
	struct pcie_port	pp;
};

#define PCIE_CH0				0
#define PCIE_CH1				1

/* PCIe STU registers */
#define PCIE_IRQ_PULSE			0x000
#define IRQ_INTA_ASSERT			(0x1 << 0)
#define IRQ_INTB_ASSERT			(0x1 << 2)
#define IRQ_INTC_ASSERT			(0x1 << 4)
#define IRQ_INTD_ASSERT			(0x1 << 6)
#define PCIE_IRQ_LEVEL			0x004
#define PCIE_IRQ_SPECIAL		0x008
#define PCIE_IRQ_EN_PULSE		0x00c
#define PCIE_IRQ_EN_LEVEL		0x010
#define IRQ_MSI_ENABLE			(0x1 << 2)
#define PCIE_IRQ_EN_SPECIAL		0x014
#define PCIE_PWR_RESET			0x018
#define PCIE_CORE_RESET			0x01c
#define PCIE_CORE_RESET_ENABLE		(0x1 << 0)
#define PCIE_STICKY_RESET		0x020
#define PCIE_NONSTICKY_RESET		0x024
#define PCIE_APP_INIT_RESET		0x028
#define PCIE_APP_LTSSM_ENABLE		0x02c
#define PCIE_STU_RDLH_LINKUP		0x064
#define PCIE_STU_LTSSM_ENABLE		0x1
#define PCIE_STU_SLV_AWMISC		0x11c
#define PCIE_STU_SLV_ARMISC		0x120
#define PCIE_STU_SLV_DBI_ENABLE	(0x1 << 21)



#define PCIE0_INT_EN1			0x118
#define PCIE0_MSI_CTRL_INT_EN			(0x1 << 0)

#define PCIE1_INT_EN1			0x118
#define PCIE1_MSI_CTRL_INT_EN			(0x1 << 0)

#define PCIE0_INT0				0x124
#define PCIE0_MSI_CTRL_INT				(0x1 << 0)

#define PCIE1_INT0				0x124
#define PCIE1_MSI_CTRL_INT				(0x1 << 0)

/************************************** PCIEx_reg 0x7C30CXXX **************************************/
#define PCIE0_REG_SII_DEBUG							0x464
#define PCIE1_REG_SII_DEBUG							0x764

#define PCIE_RDLH_LINK_UP							(0x1 << 8)

#define PCIE0_REG_PCIE0_PHY_COMMON_INTERFACE1		0x500
#define PCIE0_REF_RATE								(0x1 << 4)
#define PCIE0_MAC_PHYSTATUS							(0xF << 24)

#define PCIE1_REG_PCIE1_PHY_COMMON_INTERFACE1		0x800
#define PCIE1_REF_RATE								(0x1 << 4)
#define PCIE1_MAC_PHYSTATUS							(0x1 << 24)


#define PCIE0_REG_PCIE0_SII_POWER_MANAGEMENT		0x4AC
//0x00000001UL
#define PCIE0_SYS_AUX_PWR_DET						(0x1 << 0)

#define PCIE1_REG_PCIE1_SII_POWER_MANAGEMENT		0x7AC
#define PCIE1_SYS_AUX_PWR_DET						(0x1 << 0)


#define PCIE0_REG_PCIE0_CLOCK_AND_RESET				0x000
//0x0000009DUL
#define PCIE0_APP_CLK_REQ_N							(0x1 << 0)
#define PCIE0_PERST_N								(0x1 << 2)
#define PCIE0_CFG_L1_AUX_CLK_SWITCH_CORE_CLK_GATE_EN	(0x1 << 3)
#define PCIE0_POWER_UP_RST_N						(0x1 << 4)
#define PCIE0_SUB_RST_N								(0x1 << 7)

#define PCIE1_REG_PCIE1_CLOCK_AND_RESET				0x004

#define PCIE1_APP_CLK_REQ_N							(0x1 << 0)
#define PCIE1_PERST_N								(0x1 << 2)
#define PCIE1_CFG_L1_AUX_CLK_SWITCH_CORE_CLK_GATE_EN	(0x1 << 3)
#define PCIE1_POWER_UP_RST_N						(0x1 << 4)
#define PCIE1_SUB_RST_N								(0x1 << 7)


#define PCIE0_REG_PCIE0_SII_GENERAL_CORE_CONTROL	0x47C
// = 0x00000041UL;					
#define PCIE0_APP_LTSSM_ENABLE						(0x1 << 0)
#define PCIE0_DEVICE_TYPE							(0x4 << 4)

#define PCIE1_REG_PCIE1_SII_GENERAL_CORE_CONTROL	0x77C

#define PCIE1_APP_LTSSM_ENABLE						(0x1 << 0)
#define PCIE1_DEVICE_TYPE							(0x4 << 4)


#define PCIE0_AXI_SLAVE_INTERFACE_W					0x400
#define PCIE0_AXI_SLAVE_INTERFACE_R					0x404

#define PCIE1_AXI_SLAVE_INTERFACE_W					0x700
#define PCIE1_AXI_SLAVE_INTERFACE_R					0x704

/**************************************************************************************************/

/* PCIe Purple registers */
#define PCIE_PHY_GLOBAL_RESET		0x000
#define PCIE_PHY_COMMON_RESET		0x004
#define PCIE_PHY_CMN_REG		0x008
#define PCIE_PHY_MAC_RESET		0x00c
#define PCIE_PHY_PLL_LOCKED		0x010
#define PCIE_PHY_TRSVREG_RESET		0x020
#define PCIE_PHY_TRSV_RESET		0x024

/****************************************** CKG 0x7C015XXX *****************************************/

#define CKG_CLKSTOP_STU					0x090

#define PCIE_STOP_PCIE_AUX				(0x1 << 13)
#define PCIE_STOP_AXI_ST_PCIE			(0x1 << 4)

#define PCIE_STOP_PCIE4_AUX				(0x1 << 12)
#define PCIE_STOP_AXI_ST_PCIE4			(0x1 << 3)
#define CKG_CLKSTOP_PHY_PCIE			0x09C
#define PCIE_STOP_PHY_PCIE4				(0x1 << 1)
#define PCIE_STOP_PHY_PCIE				(0x1 << 0)

#define CKG_EXTCLK_MODE					0x22C

#define CKG_SFTRST4						0x1E0
#define PCIE_SFTRST_STU_AHB_STUREG		(0x1 << 16)
#define PCIE_SFTRST_PCIE4_REF			(0x1 << 24)
#define PCIE_SFTRST_PCIE_APB			(0x1 << 23)
#define PCIE_SFTRST_PCIE_REF			(0x1 << 22)

#define CKG_SFTRST5						0x1E4
#define PCIE_SFTRST_STU_AXI_ST_PCIE		(0x1 << 6)
#define PCIE_SFTRST_STU_AXI_ST_PCIE4	(0x1 << 5)
#define PCIE_SFTRST_PCIE4_APB			(0x1 << 2)
#define PCIE_SFTRST_PCIE4_CMN			(0x1 << 1)
#define PCIE_SFTRST_PCIE_CMN			(0x1 << 4)
/**************************************************************************************************/


/* PCIe PHY registers */
#define PCIE_PHY_IMPEDANCE		0x004
#define PCIE_PHY_PLL_DIV_0		0x008
#define PCIE_PHY_PLL_BIAS		0x00c
#define PCIE_PHY_DCC_FEEDBACK		0x014
#define STOP_PHY_DCC_CALIB              0x29

#define PCIE_PHY_PLL_DIV_1		0x05c
#define PCIE_PHY_COMMON_POWER		0x064
#define PCIE_PHY_COMMON_PD_CMN		(0x1 << 3)
#define PCIE_PHY_TRSV0_EMP_LVL		0x084
#define PCIE_PHY_TRSV0_DRV_LVL		0x088
#define PCIE_PHY_TRSV0_RXCDR		0x0ac
#define PCIE_PHY_TRSV0_POWER		0x0c4
#define PCIE_PHY_TRSV0_PD_TSV		(0x1 << 7)
#define PCIE_PHY_TRSV0_LVCC		0x0dc
#define PCIE_PHY_TRSV1_EMP_LVL		0x144
#define PCIE_PHY_TRSV1_RXCDR		0x16c
#define PCIE_PHY_TRSV1_POWER		0x184
#define PCIE_PHY_TRSV1_PD_TSV		(0x1 << 7)
#define PCIE_PHY_TRSV1_LVCC		0x19c
#define PCIE_PHY_TRSV2_EMP_LVL		0x204
#define PCIE_PHY_TRSV2_RXCDR		0x22c
#define PCIE_PHY_TRSV2_POWER		0x244
#define PCIE_PHY_TRSV2_PD_TSV		(0x1 << 7)
#define PCIE_PHY_TRSV2_LVCC		0x25c
#define PCIE_PHY_TRSV3_EMP_LVL		0x2c4
#define PCIE_PHY_TRSV3_RXCDR		0x2ec
#define PCIE_PHY_TRSV3_POWER		0x304
#define PCIE_PHY_TRSV3_PD_TSV		(0x1 << 7)
#define PCIE_PHY_TRSV3_LVCC		0x31c

/* Synopsis specific PCIE configuration registers */
#define PCIE_ATU_BUS(x)			(((x) & 0xff) << 24)
#define PCIE_ATU_DEV(x)			(((x) & 0x1f) << 19)
#define PCIE_ATU_FUNC(x)		(((x) & 0x7) << 16)


#define PVC03V_PCIE0_VENDOR_ID		0x1189
#define PVC03V_PCIE0_DEVICE_ID		0xABCD

#define PCIE_LINK_WIDTH_SPEED_CONTROL   0x80C
#define PORT_LOGIC_SPEED_CHANGE         (0x1 << 17)

#define SUPPORT_DSC		1
#if SUPPORT_DSC
/* waiting time */
#define PCIE0_STARTUP_WAITTIME		1
#define PCIE0_LINKUP_WAITTIME		10

/* timeout count */
#define PCIE0_STARTUP_TIMEOUT_CNT	50
#define PCIE0_LINKUP_TIMEOUT_CNT	50
#if SUPPORT_DSC
#define PCIE0_LINKUP_TIMEOUT_CNT_PON	10
#else
#define PCIE0_LINKUP_TIMEOUT_CNT_PON	PCIE0_LINKUP_TIMEOUT_CNT
#endif
/* reg get 0x7C30C47C */
#define PCIE0_REG_EJECT_CARD		0x00000040
extern u32 pvc03v_pcie0_reg_get_0x7C30C47C(void);

/* insertion and removal. */
#define PCIE0_CARD_REMOVE_STRONG_ON 1
#if PCIE0_CARD_REMOVE_STRONG_ON
extern u32 pvc03v_pcie0_chk_card_remove(void);
static u32 g_limit_reg_access = 0;	        //1:WX^ANZXɐA0:Ȃ
static u32 g_linkup_poweron_first = 0;		//1:dONlinkupĂA0:linkupĂȂ
/* limit to access reg. */
#define PCIE0_CARD_REMOVE_NONEACCESS_STU 1
#define PCIE0_CARD_REMOVE_NONEACCESS_PHY 0
#define PCIE0_CARD_REMOVE_NONEACCESS_BLK 0	
#define PCIE0_CARD_REMOVE_NONEACCESS_DBI 1
#endif /*PCIE0_CARD_REMOVE_STRONG_ON*/
#endif /*SUPPORT_DSC*/

/* add PCIE0 power on/off interface*/
static struct pvc03v_pcie *pvc03v_backup = NULL;

static int pvc03v_pcie0_set_gen2x1(void);
#if SUPPORT_DSC
u32 g_de_emphasis_media_35 = 0;
extern void pvc03v_pcie0_set_compliance_de_emphasis(void);
#endif
static inline void pvc03v_stu_writel(struct pvc03v_pcie *pcie, u32 val, u32 reg)
{
#if PCIE0_CARD_REMOVE_NONEACCESS_STU
	if(pvc03v_pcie0_chk_card_remove() == 0)
#endif
	{
	writel(val, pcie->stu_base + reg);
	}
}

static inline u32 pvc03v_stu_readl(struct pvc03v_pcie *pcie, u32 reg)
{
#if PCIE0_CARD_REMOVE_NONEACCESS_STU
	if((reg == 0x47c) || (reg == 0x494)){
		return readl(pcie->stu_base + reg);
	}
	if((pvc03v_pcie0_chk_card_remove() == 0))
#else
	if(1)
#endif
	{
	return readl(pcie->stu_base + reg);
	}else{
		return 0;
	}
}

static inline void pvc03v_phy_writel(struct pvc03v_pcie *pcie, u32 val, u32 reg)
{
#if PCIE0_CARD_REMOVE_NONEACCESS_PHY
	if(pvc03v_pcie0_chk_card_remove() == 0)
#endif
	{
	writel(val, pcie->phy_base + reg);
	}
}

static inline u32 pvc03v_phy_readl(struct pvc03v_pcie *pcie, u32 reg)
{
#if PCIE0_CARD_REMOVE_NONEACCESS_PHY
	if(pvc03v_pcie0_chk_card_remove() == 0)
#else
	if(1)
#endif
	{
	return readl(pcie->phy_base + reg);
	}else{
		return 0;
	}
}

static inline void pvc03v_blk_writel(struct pvc03v_pcie *pcie, u32 val, u32 reg)
{
#if PCIE0_CARD_REMOVE_NONEACCESS_BLK
	if(pvc03v_pcie0_chk_card_remove() == 0)
#endif
	{
	writel(val, pcie->block_base + reg);
	}
}

static inline u32 pvc03v_blk_readl(struct pvc03v_pcie *pcie, u32 reg)
{
#if PCIE0_CARD_REMOVE_NONEACCESS_BLK
	if(pvc03v_pcie0_chk_card_remove() == 0)
#else
	if(1)
#endif
	{
	return readl(pcie->block_base + reg);
	}else{
		return 0;
	}
}


static void pvc03v_pcie_sideband_dbi_w_mode(struct pcie_port *pp, bool on)
{
}

static void pvc03v_pcie_sideband_dbi_r_mode(struct pcie_port *pp, bool on)
{
}

static void pvc03v_pcie_assert_core_reset(struct pcie_port *pp)
{
}

static void pvc03v_pcie_deassert_core_reset(struct pcie_port *pp)
{
}

static void pvc03v_pcie_assert_phy_reset(struct pcie_port *pp)
{
}

static void pvc03v_pcie_deassert_phy_reset(struct pcie_port *pp)
{
}

static void pvc03v_pcie_power_on_phy(struct pcie_port *pp)
{
}

static void pvc03v_pcie_power_off_phy(struct pcie_port *pp)
{
}

static void pvc03v_pcie_init_phy(struct pcie_port *pp)
{
}

static void pvc03v_pcie_assert_reset(struct pcie_port *pp)
{
}


static int pvc03v_pcie_establish_link(struct pcie_port *pp)
{
	u32 val;
	int count = 0;
	struct pvc03v_pcie *pvc03v_pcie = to_pvc03v_pcie(pp);
	unsigned int	uiPsw;

	if (dw_pcie_link_up(pp)) {
		dev_err(pp->dev, "Link already up\n");
		return 0;
	}


/** for Ch0 *********************************************************************************/
    if (pvc03v_pcie->pcie_ch == PCIE_CH0) {

	DRV_CMN_SPIN_LOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);		/* rJn 								*/

    val = pvc03v_blk_readl(pvc03v_pcie, CKG_CLKSTOP_STU);
	val &= ~(PCIE_STOP_PCIE4_AUX | PCIE_STOP_AXI_ST_PCIE4);
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_CLKSTOP_STU);
	
	val = pvc03v_blk_readl(pvc03v_pcie, CKG_CLKSTOP_PHY_PCIE);
	val |= PCIE_STOP_PHY_PCIE4;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_CLKSTOP_PHY_PCIE);
	
	val = pvc03v_blk_readl(pvc03v_pcie, CKG_CLKSTOP_PHY_PCIE);
	val &= ~(PCIE_STOP_PHY_PCIE4);
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_CLKSTOP_PHY_PCIE);
	
	val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST4);
	val |= PCIE_SFTRST_STU_AHB_STUREG;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST4);
	
	val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST5);
	val |= PCIE_SFTRST_STU_AXI_ST_PCIE4;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST5);

	DRV_CMN_SPIN_UNLOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);	/* rI 								*/

	pvc03v_stu_writel(pvc03v_pcie, PCIE0_REF_RATE, PCIE0_REG_PCIE0_PHY_COMMON_INTERFACE1);

   	DRV_CMN_SPIN_LOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);		/* rJn 								*/

	val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST5);
	val |= PCIE_SFTRST_PCIE4_APB;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST5);

	val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST4);
	val |= PCIE_SFTRST_PCIE4_REF;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST4);

    DRV_CMN_SPIN_UNLOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);	/* rI 								*/
	
printk(KERN_ERR "[PCI] %s PCIE_PHY_DCC_FEEDBACK : 0x%08x\n",__func__, pvc03v_phy_readl(pvc03v_pcie, PCIE_PHY_DCC_FEEDBACK));
	/* STOP Phy DCC Calibration */
	pvc03v_phy_writel(pvc03v_pcie, STOP_PHY_DCC_CALIB, PCIE_PHY_DCC_FEEDBACK);	
printk(KERN_ERR "[PCI] %s PCIE_PHY_DCC_FEEDBACK : 0x%08x\n",__func__, pvc03v_phy_readl(pvc03v_pcie, PCIE_PHY_DCC_FEEDBACK));

	
	pvc03v_stu_writel(pvc03v_pcie, PCIE0_SYS_AUX_PWR_DET, PCIE0_REG_PCIE0_SII_POWER_MANAGEMENT);

	pvc03v_stu_writel(pvc03v_pcie, 
		(PCIE0_SUB_RST_N | PCIE0_POWER_UP_RST_N | PCIE0_CFG_L1_AUX_CLK_SWITCH_CORE_CLK_GATE_EN | PCIE0_PERST_N | PCIE0_APP_CLK_REQ_N), 
		PCIE0_REG_PCIE0_CLOCK_AND_RESET);

	DRV_CMN_SPIN_LOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);		/* rJn 								*/

    val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST5);
	val |= PCIE_SFTRST_PCIE4_CMN;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST5);
		
	DRV_CMN_SPIN_UNLOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);	/* rI 								*/

	/* wait startup*/
	while ((PCIE0_MAC_PHYSTATUS & pvc03v_stu_readl(pvc03v_pcie, PCIE0_REG_PCIE0_PHY_COMMON_INTERFACE1)) != 0) {
#if (!SUPPORT_DSC)
		mdelay(100);
#else
		mdelay(PCIE0_STARTUP_WAITTIME);
#endif
		count++;
#if (!SUPPORT_DSC)
		if (count == 50) {
#else
		if (count == PCIE0_STARTUP_TIMEOUT_CNT) {
#endif
			val = pvc03v_stu_readl(pvc03v_pcie, PCIE0_REG_PCIE0_PHY_COMMON_INTERFACE1);
			printk(KERN_ERR "[PCI] %s PCIE0_REG_PCIE0_PHY_COMMON_INTERFACE1 : 0x%08x\n",__func__, val);
			dev_err(pp->dev, "PCIe Startup Fail\n");
			return -EINVAL;
		}
	}
	
	/* setup root complex */
	dw_pcie_setup_rc(pp);

/*setting pcie root information*/
        dw_pcie_cfg_write(pp->dbi_base, PCI_VENDOR_ID, 2, PVC03V_PCIE0_VENDOR_ID);
        dw_pcie_cfg_write(pp->dbi_base, PCI_DEVICE_ID, 2, PVC03V_PCIE0_DEVICE_ID);
#if SUPPORT_DSC
		pvc03v_pcie0_set_compliance_de_emphasis();
#endif
	pvc03v_stu_writel(pvc03v_pcie, (PCIE0_APP_LTSSM_ENABLE | PCIE0_DEVICE_TYPE),
			  PCIE0_REG_PCIE0_SII_GENERAL_CORE_CONTROL);
	
/********************************************************************************************/
	}
	else {
		return 0;
	}
#if SUPPORT_DSC
	count = 0; /* inited count */
#endif
		
	/* check if the link is up or not */
	while (!dw_pcie_link_up(pp)) {
#if (!SUPPORT_DSC)
		mdelay(100);
#else
		mdelay(PCIE0_LINKUP_WAITTIME);
#endif
		count++;
#if (!SUPPORT_DSC)
		if (count == 10) {
#else
		if ((pvc03v_pcie0_reg_get_0x7C30C47C() == PCIE0_REG_EJECT_CARD) ||
			(count >= PCIE0_LINKUP_TIMEOUT_CNT_PON)) {			
#endif
printk(KERN_ERR "[PCI] %s wait link up time over\n",__func__);
/* The following code is described in OSS. */
/* Since it is processing for debug, the operation of PCI driver is not affected. */
			while (pvc03v_phy_readl(pvc03v_pcie,
						PCIE_PHY_PLL_LOCKED) == 0) {
				val = pvc03v_blk_readl(pvc03v_pcie,
						       PCIE_PHY_PLL_LOCKED);
				dev_info(pp->dev, "PLL Locked: 0x%x\n", val);
			}
			/* power off phy */
			pvc03v_pcie_power_off_phy(pp);

			dev_err(pp->dev, "PCIe Link Fail\n");
			return -EINVAL;
		}
	}

	dev_info(pp->dev, "Link up\n");

#if PCIE0_CARD_REMOVE_STRONG_ON
	/* NɁAlinkup */
	g_linkup_poweron_first = 1;
#endif

	return 0;
}

static void pvc03v_pcie_clear_irq_pulse(struct pcie_port *pp)
{
	u32 val;
	struct pvc03v_pcie *pvc03v_pcie = to_pvc03v_pcie(pp);

	val = pvc03v_stu_readl(pvc03v_pcie, PCIE_IRQ_PULSE);
	pvc03v_stu_writel(pvc03v_pcie, val, PCIE_IRQ_PULSE);
	return;
}

static void pvc03v_pcie_enable_irq_pulse(struct pcie_port *pp)
{
	u32 val;
	struct pvc03v_pcie *pvc03v_pcie = to_pvc03v_pcie(pp);

	/* enable INTX interrupt */
	val = IRQ_INTA_ASSERT | IRQ_INTB_ASSERT |
		IRQ_INTC_ASSERT | IRQ_INTD_ASSERT,
	pvc03v_stu_writel(pvc03v_pcie, val, PCIE_IRQ_EN_PULSE);
	return;
}

static irqreturn_t pvc03v_pcie_irq_handler(int irq, void *arg)
{
	struct pcie_port *pp = arg;

	pvc03v_pcie_clear_irq_pulse(pp);
	return IRQ_HANDLED;
}

static irqreturn_t pvc03v_pcie_msi_irq_handler(int irq, void *arg)
{
	struct pcie_port *pp = arg;
	struct pvc03v_pcie *pvc03v_pcie = to_pvc03v_pcie(pp);

	
	pvc03v_stu_writel(pvc03v_pcie, PCIE0_MSI_CTRL_INT, PCIE0_INT0);
	
	return dw_handle_msi_irq(pp);
}

static void pvc03v_pcie_msi_init(struct pcie_port *pp)
{
	u32 val;
	struct pvc03v_pcie *pvc03v_pcie = to_pvc03v_pcie(pp);

	dw_pcie_msi_init(pp);

#if PCIE0_CARD_REMOVE_STRONG_ON
	/* NMIG[ɂȂ邽߁AŗLɂBANɃJ[h}ꍇ̂ */
	if(g_linkup_poweron_first == 1){
		g_limit_reg_access = 1;
	}
#endif	

	val = pvc03v_stu_readl(pvc03v_pcie, PCIE0_INT_EN1);
	val |= PCIE0_MSI_CTRL_INT_EN;
	pvc03v_stu_writel(pvc03v_pcie, val, PCIE0_INT_EN1);
	
	return;
}

static void pvc03v_pcie_enable_interrupts(struct pcie_port *pp)
{
	pvc03v_pcie_enable_irq_pulse(pp);

	if (IS_ENABLED(CONFIG_PCI_MSI))
		pvc03v_pcie_msi_init(pp);

	return;
}

static inline void pvc03v_pcie_readl_rc(struct pcie_port *pp,
					void __iomem *dbi_base, u32 *val)
{
#if PCIE0_CARD_REMOVE_NONEACCESS_DBI
	if(pvc03v_pcie0_chk_card_remove() == 0)
#endif
	{
	*val = readl(dbi_base);
	}

	return;
}

static inline void pvc03v_pcie_writel_rc(struct pcie_port *pp,
					u32 val, void __iomem *dbi_base)
{
#if PCIE0_CARD_REMOVE_NONEACCESS_DBI
	if(pvc03v_pcie0_chk_card_remove() == 0)
#endif
	{
	writel(val, dbi_base);
	}
	return;
}

/**************************************************************************************************************/
static int pvc03v_pcie_rd_own_conf(struct pcie_port *pp, int where, int size,
				u32 *val)
{
	int ret;

	ret = dw_pcie_cfg_read(pp->dbi_base + (where & ~0x3), where, size, val);

	return ret;
}

static int pvc03v_pcie_wr_own_conf(struct pcie_port *pp, int where, int size,
				u32 val)
{
	int ret;

	ret = dw_pcie_cfg_write(pp->dbi_base + (where & ~0x3),
			where, size, val);

	return ret;
}

static int pvc03v_pcie_link_up(struct pcie_port *pp)
{
	struct pvc03v_pcie *pvc03v_pcie = to_pvc03v_pcie(pp);

	u32 val;
	
	if (pvc03v_pcie->pcie_ch == PCIE_CH0){
		val = pvc03v_stu_readl(pvc03v_pcie, PCIE0_REG_SII_DEBUG);
	}
	else {
		return 0;
	}
	if ((val & PCIE_RDLH_LINK_UP) == PCIE_RDLH_LINK_UP) {
		return 1;
	}
	
	return 0;
}

static void pvc03v_pcie_host_init(struct pcie_port *pp)
{
	pvc03v_pcie_establish_link(pp);
	pvc03v_pcie_enable_interrupts(pp);

}

static struct pcie_host_ops pvc03v_pcie_host_ops = {
	.readl_rc = pvc03v_pcie_readl_rc,
	.writel_rc = pvc03v_pcie_writel_rc,
	.rd_own_conf = pvc03v_pcie_rd_own_conf,
	.wr_own_conf = pvc03v_pcie_wr_own_conf,
	.link_up = pvc03v_pcie_link_up,
	.host_init = pvc03v_pcie_host_init,
};

static int __init add_pcie_port(struct pcie_port *pp,
				struct platform_device *pdev)
{
	int ret;

	pp->irq = platform_get_irq(pdev, 1);
	if (!pp->irq) {
		dev_err(&pdev->dev, "failed to get irq\n");
		return -ENODEV;
	}
	ret = devm_request_irq(&pdev->dev, pp->irq, pvc03v_pcie_irq_handler,
				IRQF_SHARED, "pvc03v-pcie", pp);
	if (ret) {
		dev_err(&pdev->dev, "failed to request irq\n");
		return ret;
	}

	if (IS_ENABLED(CONFIG_PCI_MSI)) {
		pp->msi_irq = platform_get_irq(pdev, 0);
		if (!pp->msi_irq) {
			dev_err(&pdev->dev, "failed to get msi irq\n");
			return -ENODEV;
		}

		ret = devm_request_irq(&pdev->dev, pp->msi_irq,
					pvc03v_pcie_msi_irq_handler,
					IRQF_SHARED, "pvc03v-pcie", pp);
		if (ret) {
			dev_err(&pdev->dev, "failed to request msi irq\n");
			return ret;
		}
	}

	pp->root_bus_nr = -1;
	pp->ops = &pvc03v_pcie_host_ops;

	ret = dw_pcie_host_init(pp);
	if (ret) {
		dev_err(&pdev->dev, "failed to initialize host\n");
		return ret;
	}

	return 0;
}

static int __init pvc03v_pcie_probe(struct platform_device *pdev)
{
	struct pvc03v_pcie *pvc03v_pcie;
	struct pcie_port *pp;
	struct device_node *np = pdev->dev.of_node;
	struct resource *stu_base;
	struct resource *phy_base;
	struct resource *block_base;
	struct resource *dbi_base;
	int ret;
	
	pvc03v_pcie = devm_kzalloc(&pdev->dev, sizeof(*pvc03v_pcie),
				GFP_KERNEL);
	if (!pvc03v_pcie)
		return -ENOMEM;
	/* for PCIE0 power on/off interface backup struct*/
	pvc03v_backup = pvc03v_pcie;

	pp = &pvc03v_pcie->pp;

	pp->dev = &pdev->dev;

	pvc03v_pcie->reset_gpio = of_get_named_gpio(np, "reset-gpio", 0);

	pvc03v_pcie->clk = devm_clk_get(&pdev->dev, "pcie");
	if (IS_ERR(pvc03v_pcie->clk)) {
		dev_err(&pdev->dev, "Failed to get pcie rc clock\n");
		return PTR_ERR(pvc03v_pcie->clk);
	}
	ret = clk_prepare_enable(pvc03v_pcie->clk);
	if (ret)
		return ret;

	pvc03v_pcie->bus_clk = devm_clk_get(&pdev->dev, "pcie_bus");
	if (IS_ERR(pvc03v_pcie->bus_clk)) {
		dev_err(&pdev->dev, "Failed to get pcie bus clock\n");
		ret = PTR_ERR(pvc03v_pcie->bus_clk);
		goto fail_clk;
	}
	ret = clk_prepare_enable(pvc03v_pcie->bus_clk);
	if (ret)
		goto fail_clk;

	stu_base = platform_get_resource(pdev, IORESOURCE_MEM, 0);

	if (stu_base->start == 0x7C30C000) {
		pvc03v_pcie->pcie_ch = PCIE_CH0;
	}
	else {
printk(KERN_ERR "[PCI] %s PCIe Invalid ch\n",__func__, pvc03v_pcie->pcie_ch);
	}

	pvc03v_pcie->stu_base = devm_ioremap_resource(&pdev->dev, stu_base);
	if (IS_ERR(pvc03v_pcie->stu_base)) {
		dev_err(&pdev->dev, "Failed stu_base resource\n");
		ret = PTR_ERR(pvc03v_pcie->stu_base);
		goto fail_bus_clk;
	}

	
	
	phy_base = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	pvc03v_pcie->phy_base = devm_ioremap_resource(&pdev->dev, phy_base);

	if (IS_ERR(pvc03v_pcie->phy_base)) {
		dev_err(&pdev->dev, "Failed phy_base resource\n");
		ret = PTR_ERR(pvc03v_pcie->phy_base);
		goto fail_bus_clk;
	}

	block_base = platform_get_resource(pdev, IORESOURCE_MEM, 2);
	pvc03v_pcie->block_base = devm_ioremap_resource(&pdev->dev, block_base);

	if (IS_ERR(pvc03v_pcie->block_base)) {
		dev_err(&pdev->dev, "Failed block_base resource\n");
		ret = PTR_ERR(pvc03v_pcie->block_base);
		goto fail_bus_clk;
	}

	/*add dbi_base initialize*/
	dbi_base = platform_get_resource(pdev, IORESOURCE_MEM, 3);
	pp->dbi_base = devm_ioremap_resource(&pdev->dev, dbi_base);

	if (IS_ERR(pp->dbi_base)) {
		dev_err(&pdev->dev, "Failed block_base resource\n");
		ret = PTR_ERR(pp->dbi_base);
		goto fail_bus_clk;
	}

	ret = add_pcie_port(pp, pdev);
	if (ret < 0)
		goto fail_bus_clk;

	platform_set_drvdata(pdev, pvc03v_pcie);
	return 0;

fail_bus_clk:
	clk_disable_unprepare(pvc03v_pcie->bus_clk);
fail_clk:
	clk_disable_unprepare(pvc03v_pcie->clk);
	return ret;
}

static int __exit pvc03v_pcie_remove(struct platform_device *pdev)
{
	struct pvc03v_pcie *pvc03v_pcie = platform_get_drvdata(pdev);

	clk_disable_unprepare(pvc03v_pcie->bus_clk);
	clk_disable_unprepare(pvc03v_pcie->clk);

	return 0;
}

static const struct of_device_id pvc03v_pcie_of_match[] = {
	{ .compatible = "panasonic,pvc03v-pcie", },
	{},
};
MODULE_DEVICE_TABLE(of, pvc03v_pcie_of_match);

static struct platform_driver pvc03v_pcie_driver = {
	.remove		= __exit_p(pvc03v_pcie_remove),
#if 0
	.probe		= pvc03v_pcie_probe,
#endif
	.driver = {
		.name	= "pvc03v-pcie",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(pvc03v_pcie_of_match),
	},
};

/* pvc03v PCIe driver does not allow module unload */
static int __init pcie_init(void)
{
	int ret;

	ret = platform_driver_probe(&pvc03v_pcie_driver, pvc03v_pcie_probe);
//	ret = platform_driver_register(&pvc03v_pcie_driver);

	return ret;
}
subsys_initcall(pcie_init);

static int pvc03v_pcie0_set_gen2x1(void)
{
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
	struct pcie_port *pp = &pvc03v_pcie->pp;
	int ret;
	u32 iomem_value = 0x00000000;

	/* offset is  0x0710 */
	ret = dw_pcie_cfg_read(pp->dbi_base + 0x0710, 0x0710, 4, &iomem_value);
	if(ret != PCIBIOS_SUCCESSFUL){
		return -EINVAL;
	}
	iomem_value = (iomem_value & 0xffc0ffff) | 0x00010000;
	dw_pcie_cfg_write(pp->dbi_base + 0x0710, 0x0710, 4, iomem_value);
	ret = dw_pcie_cfg_read(pp->dbi_base + 0x0710, 0x0710, 4, &iomem_value);
	if(ret != PCIBIOS_SUCCESSFUL){
	    return -EINVAL;
	}
	
	/* offset is  0x080c */
	ret = dw_pcie_cfg_read(pp->dbi_base + 0x080c, 0x080c, 4, &iomem_value);
	if(ret != PCIBIOS_SUCCESSFUL){
	    return -EINVAL;
	}
	iomem_value = (iomem_value & 0xffff00ff) | 0x00000100;
	dw_pcie_cfg_write(pp->dbi_base + 0x080c, 0x080c, 4, iomem_value);
	ret = dw_pcie_cfg_read(pp->dbi_base + 0x080c, 0x080c, 4, &iomem_value);
	if(ret != PCIBIOS_SUCCESSFUL){
	    return -EINVAL;
	}

	/* offset is  0x007c */
	ret = dw_pcie_cfg_read(pp->dbi_base + 0x007c, 0x007c, 4, &iomem_value);
	if(ret != PCIBIOS_SUCCESSFUL){
	    return -EINVAL;
	}
	iomem_value = (iomem_value & 0xffffff0f) | 0x00000010;
	dw_pcie_cfg_write(pp->dbi_base + 0x007c, 0x007c, 4, iomem_value);
	ret = dw_pcie_cfg_read(pp->dbi_base + 0x007c, 0x007c, 4, &iomem_value);
	if(ret != PCIBIOS_SUCCESSFUL){
	    return -EINVAL;
	}
	return 0;
}

/* add PCIE0 power on/off interface*/
int pvc03v_pcie0_phy_on(void)
{
	u32 val;
        int count = 0;
        struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
	unsigned int	uiPsw;
	
	DRV_CMN_SPIN_LOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);		/* rJn 								*/

	val = pvc03v_blk_readl(pvc03v_pcie, CKG_CLKSTOP_STU);
	val &= ~(PCIE_STOP_PCIE4_AUX | PCIE_STOP_AXI_ST_PCIE4);
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_CLKSTOP_STU);
	
	val = pvc03v_blk_readl(pvc03v_pcie, CKG_CLKSTOP_PHY_PCIE);
	val |= PCIE_STOP_PHY_PCIE4;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_CLKSTOP_PHY_PCIE);
	
	val = pvc03v_blk_readl(pvc03v_pcie, CKG_CLKSTOP_PHY_PCIE);
	val &= ~(PCIE_STOP_PHY_PCIE4);
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_CLKSTOP_PHY_PCIE);
	
	val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST4);
	val |= PCIE_SFTRST_STU_AHB_STUREG;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST4);
	
	val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST5);
	val |= PCIE_SFTRST_STU_AXI_ST_PCIE4;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST5);

	DRV_CMN_SPIN_UNLOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);	/* rI 								*/

	pvc03v_stu_writel(pvc03v_pcie, PCIE0_REF_RATE, PCIE0_REG_PCIE0_PHY_COMMON_INTERFACE1);
	
	DRV_CMN_SPIN_LOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);		/* rJn 								*/

	val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST5);
	val |= PCIE_SFTRST_PCIE4_APB;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST5);

	val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST4);
	val |= PCIE_SFTRST_PCIE4_REF;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST4);

	DRV_CMN_SPIN_UNLOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);	/* rI 								*/
	
	/* STOP Phy DCC Calibration */
	pvc03v_phy_writel(pvc03v_pcie, STOP_PHY_DCC_CALIB, PCIE_PHY_DCC_FEEDBACK);	

	
	pvc03v_stu_writel(pvc03v_pcie, PCIE0_SYS_AUX_PWR_DET, PCIE0_REG_PCIE0_SII_POWER_MANAGEMENT);

	pvc03v_stu_writel(pvc03v_pcie, 
		(PCIE0_SUB_RST_N | PCIE0_POWER_UP_RST_N | PCIE0_CFG_L1_AUX_CLK_SWITCH_CORE_CLK_GATE_EN | PCIE0_PERST_N | PCIE0_APP_CLK_REQ_N), 
		PCIE0_REG_PCIE0_CLOCK_AND_RESET);

	DRV_CMN_SPIN_LOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);		/* rJn 								*/

	val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST5);
	val |= PCIE_SFTRST_PCIE4_CMN;
	pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST5);

	DRV_CMN_SPIN_UNLOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);	/* rI 								*/
		

	/* wait startup*/
	while ((PCIE0_MAC_PHYSTATUS & pvc03v_stu_readl(pvc03v_pcie, PCIE0_REG_PCIE0_PHY_COMMON_INTERFACE1)) != 0) {
#if (!SUPPORT_DSC)
		mdelay(100);
#else
		mdelay(PCIE0_STARTUP_WAITTIME);
#endif
		count++;
#if (!SUPPORT_DSC)
		if (count == 50) {
#else
		if (count == PCIE0_STARTUP_TIMEOUT_CNT) {
#endif
			val = pvc03v_stu_readl(pvc03v_pcie, PCIE0_REG_PCIE0_PHY_COMMON_INTERFACE1);
			printk(KERN_ERR "[PCI] %s PCIE0_REG_PCIE0_PHY_COMMON_INTERFACE1 : 0x%08x\n",__func__, val);
			return -EINVAL;
		}
	}

	return 0;
	
}
EXPORT_SYMBOL(pvc03v_pcie0_phy_on);

static u32 pci_0x7c1c0820_ok = 0x8377f000;

int pvc03v_pcie0_phy_on2(void)
{
	void __iomem *io_mem;

	io_mem = ioremap_nocache(0x7f0f0000, 0x1000);
    if (!io_mem) return -ENOMEM;

	writel(0x00000029, io_mem+0x014);

    iounmap(io_mem);

	io_mem = ioremap_nocache(0x7c1c0000, 0x1000);
    if (!io_mem) return -ENOMEM;

	writel(0x00100546, io_mem+0x004);
	writel(0x00000000, io_mem+0x010);
	writel(0x00010100, io_mem+0x018);
	writel(0x0000fff0, io_mem+0x020);
	writel(0x0001016d, io_mem+0x03c);
	writel(0x000004f0, io_mem+0x80c);
	writel(pci_0x7c1c0820_ok, io_mem+0x820);

    iounmap(io_mem);

	return 0;
}
EXPORT_SYMBOL(pvc03v_pcie0_phy_on2);

#if SUPPORT_DSC
#if 1 /* eepromύXɔAύX */
int pvc03v_pcie0_update_0x7c1c0820_ok(void)
{
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
    struct pcie_port *pp = &pvc03v_pcie->pp;
	void __iomem *io_mem_EEPo_un58f0;
	int ret;
	
    ret = dw_pcie_cfg_read(pp->dbi_base+0x820, 0x820, 4, &pci_0x7c1c0820_ok);
	if(ret == PCIBIOS_SUCCESSFUL){
	    io_mem_EEPo_un58f0 = ioremap_nocache((0x0015B000), 0x200); //0x0015B1E0
		if (io_mem_EEPo_un58f0){
			writew((u16)(pci_0x7c1c0820_ok >> 16), io_mem_EEPo_un58f0 + 0x1E0);                   //upper
			writew((u16)(pci_0x7c1c0820_ok & 0x0000ffff), (io_mem_EEPo_un58f0 + 0x1E0 + 0x02));   //lower
			iounmap(io_mem_EEPo_un58f0);	
			//printk("[pvc03v_pcie0_update_0x7c1c0820_ok]OK. adr=0x%08x\n", pci_0x7c1c0820_ok);
		}else{
			printk("[pvc03v_pcie0_update_0x7c1c0820_ok]NG2. Not Update EEprom. adr=0x%08x\n", pci_0x7c1c0820_ok);
		}
	}else{
		printk("[pvc03v_pcie0_update_0x7c1c0820_ok]NG. Not get Register. ret=0x%08x\n", ret);
	}
	return ret;
}
#else
int pvc03v_pcie0_update_0x7c1c0820_ok(void)
{
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
    struct pcie_port *pp = &pvc03v_pcie->pp;
	void __iomem *io_mem_EEPf_une4c0;
	int ret;
	
    ret = dw_pcie_cfg_read(pp->dbi_base+0x820, 0x820, 4, &pci_0x7c1c0820_ok);
	if(ret == PCIBIOS_SUCCESSFUL){
	    io_mem_EEPf_une4c0 = ioremap_nocache((0x001CC000), 0x200); //0x001CC980
		if (io_mem_EEPf_une4c0){
			writew((u16)(pci_0x7c1c0820_ok >> 16), io_mem_EEPf_une4c0 + 0x980);                   //upper
			writew((u16)(pci_0x7c1c0820_ok & 0x0000ffff), (io_mem_EEPf_une4c0 + 0x980 + 0x02));   //lower
			iounmap(io_mem_EEPf_une4c0);	
			//printk("[pvc03v_pcie0_update_0x7c1c0820_ok]OK. adr=0x%08x\n", pci_0x7c1c0820_ok);
		}else{
			printk("[pvc03v_pcie0_update_0x7c1c0820_ok]NG2. Not Update EEprom. adr=0x%08x\n", pci_0x7c1c0820_ok);
		}
	}else{
		printk("[pvc03v_pcie0_update_0x7c1c0820_ok]NG. Not get Register. ret=0x%08x\n", ret);
	}
	return ret;
}
#endif
EXPORT_SYMBOL(pvc03v_pcie0_update_0x7c1c0820_ok);
#endif /*SUPPORT_DSC*/

void pvc03v_pcie0_phy_off(void)
{
	
        u32 val;
        int count = 0;
	void __iomem *io_mem;
        struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;	
	unsigned int	uiPsw;

	io_mem = ioremap_nocache(0x7C160000, 0x1000);
        if (!io_mem )
                return -ENOMEM;

	DRV_CMN_SPIN_LOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);		/* rJn 								*/
        val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST4);
        val |= PCIE_SFTRST_PCIE4_REF;
        pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST4);

        val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST5);
        val |= PCIE_SFTRST_PCIE4_APB;
        pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST5);
	DRV_CMN_SPIN_UNLOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);	/* rI 								*/

	
	pvc03v_phy_writel(pvc03v_pcie, 0x00000004, 0x0008);
	pvc03v_phy_writel(pvc03v_pcie, 0x00000084, 0x0018);
	pvc03v_phy_writel(pvc03v_pcie, 0x00000000, 0x001c);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000e0, 0x0054);
	pvc03v_phy_writel(pvc03v_pcie, 0x0000004b, 0x0064);
	pvc03v_phy_writel(pvc03v_pcie, 0x00000096, 0x0084);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000c8, 0x0090);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000b5, 0x00c4);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000fc, 0x0100);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000b5, 0x0184);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000fc, 0x01c0);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000b5, 0x0244);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000fc, 0x0280);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000b5, 0x0304);
	pvc03v_phy_writel(pvc03v_pcie, 0x000000fc, 0x0340);

	DRV_CMN_SPIN_LOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);		/* rJn 								*/
        val = pvc03v_blk_readl(pvc03v_pcie, CKG_SFTRST5);
        val &= ~PCIE_SFTRST_PCIE4_CMN;
        pvc03v_blk_writel(pvc03v_pcie, val, CKG_SFTRST5);

        val = pvc03v_blk_readl(pvc03v_pcie, CKG_CLKSTOP_PHY_PCIE);
        val |= PCIE_STOP_PHY_PCIE4;
        pvc03v_blk_writel(pvc03v_pcie, val, CKG_CLKSTOP_PHY_PCIE);

        val = pvc03v_blk_readl(pvc03v_pcie, CKG_CLKSTOP_STU);
        val |= PCIE_STOP_PCIE4_AUX;
        pvc03v_blk_writel(pvc03v_pcie, val, CKG_CLKSTOP_STU);
	DRV_CMN_SPIN_UNLOCK(DRV_CMN_EXCLUSION_CKGEN, &uiPsw);	/* rI 								*/

	writel(0x00000001, io_mem + 0x104);


        while ((0x0000110 & readl(io_mem + 0x104)) == 0) {
                count++;
                if (count == 50) {
                        printk(KERN_ERR "[PCI] %s Error phy off : 0x%08x\n",__func__, readl(io_mem + 0x104));
                        return -EINVAL;
                }
        }


        while ((0x0000100 & readl(io_mem + 0x104)) == 0) {
                count++;
                if (count == 50) {
                        printk(KERN_ERR "[PCI] %s Error phy off : 0x%08x\n",__func__, readl(io_mem + 0x104));
                        return -EINVAL;
                }
        }

       pvc03v_phy_writel(pvc03v_pcie, 0x00000000, 0x01A0);

	val = readl(io_mem + 0x6C);
	val &= ~0x00000100;
        writel(val, io_mem + 0x6C);

        val = readl(io_mem + 0x6C);
        val |= 0x00000030;
        writel(val, io_mem + 0x6C);

        while ((0x00000003 & readl(io_mem + 0x6C)) != 0x00000003) {
                count++;
                if (count == 50) {
                        printk(KERN_ERR "[PCI] %s Error phy off : 0x%08x\n",__func__, readl(io_mem + 0x6C));
                        return -EINVAL;
                }
        }

        iounmap(io_mem);

#if PCIE0_CARD_REMOVE_STRONG_ON
	/* linukup finished. */
	g_limit_reg_access = 0;
#endif

	return;
	
}
EXPORT_SYMBOL(pvc03v_pcie0_phy_off);
#if SUPPORT_DSC
void pvc03v_pcie0_set_compliance_de_emphasis(void)
{
	void __iomem *io_mem;

	io_mem = ioremap_nocache(0x7F0F0000, 0x1000);
    if (!io_mem) return -ENOMEM;

	writel(0x14, io_mem+0x084);
	writel(0x14, io_mem+0x144);
	writel(0x14, io_mem+0x204);
	writel(0x14, io_mem+0x2C4);

	g_de_emphasis_media_35 = 1;

    iounmap(io_mem);
}
EXPORT_SYMBOL(pvc03v_pcie0_set_compliance_de_emphasis);

void  pvc03v_pcie0_DEBUG_reg_dump(u32 address, char *str)
{
	void __iomem *io_mem;
	u32 val;

	io_mem = ioremap_nocache(address, 4);
    if (!io_mem) return;

	val = readl(io_mem);

    iounmap(io_mem);

	printk(KERN_ERR "%s %.8x\n", str, val);
}
EXPORT_SYMBOL(pvc03v_pcie0_DEBUG_reg_dump);
#endif /*SUPPORT_DSC*/

void pvc03v_pcie0_reg_set(int lane_num)
{
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
	struct pcie_port *pp = &pvc03v_pcie->pp;

/*setting in dw_pcie_setup_rc function*/
	dw_pcie_setup_rc(pp);

/*setting pvc03v_pcie_establish_link function*/
/*setting pcie root information*/
	dw_pcie_cfg_write(pp->dbi_base, PCI_VENDOR_ID, 2, PVC03V_PCIE0_VENDOR_ID);
	dw_pcie_cfg_write(pp->dbi_base, PCI_DEVICE_ID, 2, PVC03V_PCIE0_DEVICE_ID);
	
	if(lane_num == 1){
		pvc03v_pcie0_set_gen2x1();
	}
#if SUPPORT_DSC	
	pvc03v_pcie0_set_compliance_de_emphasis();
#endif
	pvc03v_stu_writel(pvc03v_pcie, (PCIE0_APP_LTSSM_ENABLE | PCIE0_DEVICE_TYPE),
			  PCIE0_REG_PCIE0_SII_GENERAL_CORE_CONTROL);
}
EXPORT_SYMBOL(pvc03v_pcie0_reg_set);

int pvc03v_pcie0_wait_linkup(void)
{
	u32 val;
	int count = 0;
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
	struct pcie_port *pp = &pvc03v_pcie->pp;
	
	/* check if the link is up or not */
	while (!dw_pcie_link_up(pp)) {
#if (!SUPPORT_DSC)
		mdelay(100);
#else
		mdelay(PCIE0_LINKUP_WAITTIME);
#endif
		count++;
#if (!SUPPORT_DSC)
		if (count == 10) {
#else
		if ((pvc03v_pcie0_reg_get_0x7C30C47C() == PCIE0_REG_EJECT_CARD) ||
			(count >= PCIE0_LINKUP_TIMEOUT_CNT)) {
#endif
printk(KERN_ERR "[PCI] %s wait link up time over\n",__func__);
/* The following code is described in OSS. */
/* Since it is processing for debug, the operation of PCI driver is not affected. */
			while (pvc03v_phy_readl(pvc03v_pcie,
						PCIE_PHY_PLL_LOCKED) == 0) {
				val = pvc03v_blk_readl(pvc03v_pcie,
						       PCIE_PHY_PLL_LOCKED);
				dev_info(pp->dev, "PLL Locked: 0x%x\n", val);
			}
			/* power off phy */
			pvc03v_pcie_power_off_phy(pp);

			dev_err(pp->dev, "PCIe Link Fail\n");
			return -EINVAL;
		}
	}
	return 0;	
}
EXPORT_SYMBOL(pvc03v_pcie0_wait_linkup);

void pvc03v_pcie0_reg_set2(void)
{
	u32 val;	
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
	struct pcie_port *pp = &pvc03v_pcie->pp;

/*setting pvc03v_pcie_enable_irq_pulse function*/
        /* enable INTX interrupt */
        val = IRQ_INTA_ASSERT | IRQ_INTB_ASSERT |
                IRQ_INTC_ASSERT | IRQ_INTD_ASSERT,
        pvc03v_stu_writel(pvc03v_pcie, val, PCIE_IRQ_EN_PULSE);

/*setting pvc03v_pcie_msi_init function*/
        /* setting dw_pcie_msi_init function */
        dw_pcie_msi_reinit(pp);

        val = pvc03v_stu_readl(pvc03v_pcie, PCIE0_INT_EN1);
        val |= PCIE0_MSI_CTRL_INT_EN;
        pvc03v_stu_writel(pvc03v_pcie, val, PCIE0_INT_EN1);

/*setting dw_pcie_host_init function*/

        pvc03v_pcie_wr_own_conf(pp, PCI_BASE_ADDRESS_0, 4, 0);
        /* program correct class for RC */
        pvc03v_pcie_wr_own_conf(pp, PCI_CLASS_DEVICE, 2, PCI_CLASS_BRIDGE_PCI);
		
#if PCIE0_CARD_REMOVE_STRONG_ON
		/* xqddiniẗʒuɕύXBXQDJ[hԂłstrŁAJ[h}BARo^ŃG[ɂȂB */
		g_limit_reg_access = 1;
#endif
		
	return;
	
}
EXPORT_SYMBOL(pvc03v_pcie0_reg_set2);
#if SUPPORT_DSC
void pvc03v_pcie0_reg_set_0x7C30C47C(u32 val)
{
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
	if(pvc03v_pcie == NULL){
		return;
	}
	 pvc03v_stu_writel(pvc03v_pcie, val, 0x47c);
}
EXPORT_SYMBOL(pvc03v_pcie0_reg_set_0x7C30C47C);

u32 pvc03v_pcie0_reg_get_0x7C30C47C(void)
{
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
	u32 val;
	if(pvc03v_pcie == NULL){
		return 0;
	}
	val = pvc03v_stu_readl(pvc03v_pcie, 0x47c);
	return val;
}
EXPORT_SYMBOL(pvc03v_pcie0_reg_get_0x7C30C47C);

u32 pvc03v_pcie0_reg_get_0x7C30C494(void)
{
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
	u32 val;
	if(pvc03v_pcie == NULL){
		return 0;
	}
	val = pvc03v_stu_readl(pvc03v_pcie, 0x494);
	return val;
}
EXPORT_SYMBOL(pvc03v_pcie0_reg_get_0x7C30C494);

#if 1
/* PCIoX폜
   pm_suspend/resumeɁApciANZXȂ悤ɂ邽߂ɁAbus폜Ȃ
 */
extern struct pci_dev *pdev_dw_msi_setup_irq;
void  pvc03v_pcie0_remove_allbus(void)
{
	struct pci_dev *pdev = pdev_dw_msi_setup_irq;
	
	if(pdev == NULL){
		printk("pvc03v_pcie0_remove_allbus is fail. vendor_id = 0x%x, dev_id = 0x%x \n", pdev->vendor, pdev->device);
		return ;
	}
	pci_stop_and_remove_bus_device(pdev);
}
EXPORT_SYMBOL(pvc03v_pcie0_remove_allbus);
#endif

/* check card remove. */
u32 pvc03v_pcie0_chk_card_remove(void)
{
	struct pvc03v_pcie *pvc03v_pcie = pvc03v_backup;
	if((pvc03v_pcie == NULL)
#if PCIE0_CARD_REMOVE_STRONG_ON
		 || (g_limit_reg_access == 0)
#endif
	){
		return 0;
	}
	
	if(!(pvc03v_pcie0_reg_get_0x7C30C47C() & 0x00000001)){
		return 1;
	}
	if(!(pvc03v_pcie0_reg_get_0x7C30C494() & 0x00000010)){
		return 1;
	}
	
	return 0;
}
#endif
void  pvc03v_pcie0_get_id_vendor_and_device(
	unsigned short*	pci_rc_vendor,
	unsigned short*	pci_rc_device
)
{
	*pci_rc_vendor = PVC03V_PCIE0_VENDOR_ID;
	*pci_rc_device = PVC03V_PCIE0_DEVICE_ID;
}

MODULE_DESCRIPTION("Panasonic PCIe host controller driver");
MODULE_LICENSE("GPL v2");
