/*
 * Copyright (C) 2016 MediaTek Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See http://www.gnu.org/licenses/gpl-2.0.html for more details.
 */
#include <linux/init.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/sched.h>
#include <asm/current.h>
#include <asm/uaccess.h>
#include <linux/fcntl.h>
#include <linux/poll.h>
#include <linux/time.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/printk.h>

#include "osal_typedef.h"
#include "stp_exp.h"
#include "wmt_exp.h"

#if MTK_BT_HCI && REMOVE_MK_NODE
#include <net/bluetooth/bluetooth.h>
#include <net/bluetooth/hci_core.h>
#endif

MODULE_LICENSE("Dual BSD/GPL");

#define BT_DRIVER_NAME "mtk_stp_BT_chrdev"
#define BT_DEV_MAJOR 192	/* Never used number */

#define PFX                         "[MTK-BT] "
#define BT_LOG_DBG                  3
#define BT_LOG_INFO                 2
#define BT_LOG_WARN                 1
#define BT_LOG_ERR                  0

#define COMBO_IOCTL_FW_ASSERT        2
#define COMBO_IOCTL_BT_IC_HW_VER     3
#define COMBO_IOCTL_BT_IC_FW_VER     4
#define COMBO_IOC_BT_HWVER           5


static UINT32 gDbgLevel = BT_LOG_INFO;

#define BT_DBG_FUNC(fmt, arg...)	\
	do { if (gDbgLevel >= BT_LOG_DBG)	\
		pr_debug(PFX "%s: "  fmt, __func__ , ##arg);	\
	} while (0)
#define BT_INFO_FUNC(fmt, arg...)	\
do { if (gDbgLevel >= BT_LOG_INFO)	\
		pr_warn(PFX "%s: "  fmt, __func__ , ##arg);	\
	} while (0)
#define BT_WARN_FUNC(fmt, arg...)	\
	do { if (gDbgLevel >= BT_LOG_WARN)	\
		pr_err(PFX "%s: "  fmt, __func__ , ##arg);	\
	} while (0)
#define BT_ERR_FUNC(fmt, arg...)	\
	do { if (gDbgLevel >= BT_LOG_ERR)	\
		pr_err(PFX "%s: "   fmt, __func__ , ##arg);	\
	} while (0)

#define VERSION "1.0"

static INT32 BT_devs = 1;	/* Device count */
static INT32 BT_major = BT_DEV_MAJOR;	/* Dynamic allocation */
module_param(BT_major, uint, 0);
static struct cdev BT_cdev;

#define BT_BUFFER_SIZE 2048
static UINT8 i_buf[BT_BUFFER_SIZE];	/* Input buffer of read() */
static UINT8 o_buf[BT_BUFFER_SIZE];	/* Output buffer of write() */

static struct semaphore wr_mtx, rd_mtx;
/* Wait queue for poll and read */
static wait_queue_head_t inq;
static DECLARE_WAIT_QUEUE_HEAD(BT_wq);
static INT32 flag;
/* Reset flag for whole chip reset senario */
static volatile INT32 rstflag;

/* Number of running BT apps */
static atomic_t running_bts = ATOMIC_INIT(0);
/* Flag to leave BT on after unloading the driver */
MTK_WCN_BOOL leave_bt_on = MTK_WCN_BOOL_FALSE;

#if REMOVE_MK_NODE
struct class *stpbt_class = NULL;
struct device *stpbt_dev = NULL;
#endif

#if MTK_BT_HCI && REMOVE_MK_NODE
static struct mtk_hci {
	struct hci_dev *hdev;
	struct work_struct work;
	struct sk_buff_head txq;
} mtk_hci;

void
hex_dump(char *prefix, char *p, int len)
{
	int i;

	printk("%s ", prefix);
	for (i = 0; i < len; i++)
		printk("%02x ", (*p++ & 0xff));
	printk("\n");
}

static int
mtk_bt_hci_open(struct hci_dev *hdev)
{
	int err = 0;

	pr_err("# %s, return -1, do't use hci\n", __func__);
    return -1;
	/*if ((err = mtk_wcn_wmt_func_on(WMTDRV_TYPE_BT)) != MTK_WCN_BOOL_TRUE) {
		pr_err("%s func on failed with %d\n", __func__, err);
		return -ENODEV;
	}

	set_bit(HCI_RUNNING, &hdev->flags);
	set_bit(HCI_RAW, &hdev->flags);

	mtk_wcn_stp_set_bluez(1);

	return 0;*/
}

static int
mtk_bt_hci_close(struct hci_dev *hdev)
{
#if ! MTK_BLUEANGEL
	int err = 0;
#endif

	pr_err("# %s\n", __func__);

	mtk_wcn_stp_set_bluez(0);

	clear_bit(HCI_RUNNING, &hdev->flags);

#if ! MTK_BLUEANGEL
	if ((err = mtk_wcn_wmt_func_off(WMTDRV_TYPE_BT)) != MTK_WCN_BOOL_TRUE) {
		pr_err("%s func off failed with %d\n", __func__, err);
		return -EIO;
	}
#endif

	return 0;
}

static void
mtk_bt_hci_work(struct work_struct *work)
{
	int err;
	struct sk_buff *skb;

	pr_err("# %s\n", __func__);

	while ((skb = skb_dequeue(&mtk_hci.txq))) {
		skb_push(skb, 1);
		skb->data[0] = bt_cb(skb)->pkt_type;

		hex_dump(">>", skb->data, skb->len);

		if ((err = mtk_wcn_stp_send_data(skb->data, skb->len, BT_TASK_INDX)) < 0) {
			pr_err("%s err=%d\n", __func__, err);
			mtk_hci.hdev->stat.err_tx++;
			skb_queue_head(&mtk_hci.txq, skb);
			break;
		}

		mtk_hci.hdev->stat.byte_tx += skb->len;
		kfree_skb(skb);
	}
}

static int
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 11, 0)
mtk_bt_hci_send(struct hci_dev *hdev, struct sk_buff *skb)
#else
mtk_bt_hci_send(struct sk_buff *skb)
#endif
{
	pr_err("# %s\n", __func__);

	if (mtk_hci.hdev && !test_bit(HCI_RUNNING, &mtk_hci.hdev->flags))
		return -EBUSY;

	switch (bt_cb(skb)->pkt_type) {
	case HCI_COMMAND_PKT:
		mtk_hci.hdev->stat.cmd_tx++;
		break;

	case HCI_ACLDATA_PKT:
		mtk_hci.hdev->stat.acl_tx++;
		break;

	case HCI_SCODATA_PKT:
		mtk_hci.hdev->stat.sco_tx++;
		break;

	default:
		return -EILSEQ;
	}

	skb_queue_tail(&mtk_hci.txq, skb);
	schedule_work(&mtk_hci.work);

	return 0;
}

static int
mtk_bt_hci_flush(struct hci_dev *hdev)
{
	pr_err("%s\n", __func__);

	flush_work(&mtk_hci.work);

	return 0;
}

static void
mtk_bt_hci_receive(const PUINT8 data, INT32 size)
{
	int err;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 17, 0)
	struct sk_buff *skb;
#endif

	pr_err("# %s\n", __func__);

	hex_dump("<<", data, size);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 17, 0)
	if (!(skb = bt_skb_alloc(size - 1, GFP_KERNEL))) {
		pr_err("%s: bt_skb_alloc failed\n", __func__);
		return;
	}

#define MTK_IC_6630
#ifdef MTK_IC_6630
	/* temporary workaround for 3.18 with 6630 */
	if (data[0] == 0x04 &&
		data[1] == 0x0e &&
		data[2] == 0x05 &&
		data[3] == 0x01 &&
		data[4] == 0x0c &&
		data[5] == 0x14 &&
		data[6] == 0x02 &&
		data[7] == 0x00)
		data[6] = 0x00;
#endif

	skb->len = size - 1;
	bt_cb(skb)->pkt_type = data[0];
	memcpy(&skb->data[0], &data[1], size - 1);

	if ((err = hci_recv_frame(mtk_hci.hdev, skb)) < 0)
		pr_err("%s: hci_recv_frame failed with %d\n", __func__, err);
#else
	if ((err = hci_recv_fragment(mtk_hci.hdev, data[0],  (void*)&data[1], size - 1)) < 0)
		pr_err("%s: hci_recv_fragment failed with %d\n", __func__, err);
#endif

	if (mtk_hci.hdev)
		mtk_hci.hdev->stat.byte_rx += size - 1;
}

static void
mtk_bt_hci_notify(struct hci_dev *hdev, unsigned int evt)
{
	pr_info("%s event=0x%x\n", __func__, evt);
}
#endif

static VOID bt_cdev_rst_cb(ENUM_WMTDRV_TYPE_T src,
			   ENUM_WMTDRV_TYPE_T dst, ENUM_WMTMSG_TYPE_T type, PVOID buf, UINT32 sz)
{
	/*
	   Handle whole chip reset messages
	 */
	ENUM_WMTRSTMSG_TYPE_T rst_msg;

	if (sz <= sizeof(ENUM_WMTRSTMSG_TYPE_T)) {
		memcpy((PINT8)&rst_msg, (PINT8)buf, sz);
		BT_DBG_FUNC("src = %d, dst = %d, type = %d, buf = 0x%x sz = %d, max = %d\n", src,
			     dst, type, rst_msg, sz, WMTRSTMSG_RESET_MAX);
		if ((src == WMTDRV_TYPE_WMT) && (dst == WMTDRV_TYPE_BT)
		    && (type == WMTMSG_TYPE_RESET)) {
			if (rst_msg == WMTRSTMSG_RESET_START) {
				BT_INFO_FUNC("BT reset start!\n");
				rstflag = 1;
				wake_up_interruptible(&inq);

			} else if (rst_msg == WMTRSTMSG_RESET_END) {
				BT_INFO_FUNC("BT reset end!\n");
				rstflag = 2;
				wake_up_interruptible(&inq);
			}
		}
	} else {
		/* Invalid message format */
		BT_WARN_FUNC("Invalid message format!\n");
	}
}

VOID BT_event_cb(VOID)
{
	BT_DBG_FUNC("BT_event_cb()\n");

	flag = 1;

	/*
	* Finally, wake up any reader blocked in poll or read
	*/
	wake_up_interruptible(&inq);
	wake_up(&BT_wq);
}

unsigned int BT_poll(struct file *filp, poll_table *wait)
{
	UINT32 mask = 0;

/* down(&wr_mtx); */

	if (mtk_wcn_stp_is_rxqueue_empty(BT_TASK_INDX)) {
		poll_wait(filp, &inq, wait);

		if (!mtk_wcn_stp_is_rxqueue_empty(BT_TASK_INDX) || rstflag)
			/* BT Rx queue has valid data, or whole chip reset occurs */
			mask |= POLLIN | POLLRDNORM;	/* Readable */
	} else {
		mask |= POLLIN | POLLRDNORM;	/* Readable */
	}

	/* Do we need condition here? */
	mask |= POLLOUT | POLLWRNORM;	/* Writable */
/* up(&wr_mtx); */
	return mask;
}

ssize_t BT_write(struct file *filp, const char __user *buf, size_t count, loff_t *f_pos)
{
	INT32 retval = 0;
	INT32 write_size;
	INT32 written = 0;

	down(&wr_mtx);

	BT_DBG_FUNC("%s: count %zd pos %lld\n", __func__, count, *f_pos);
	if (rstflag) {
		if (rstflag == 1) {	/* Reset start */
			retval = -88;
			BT_INFO_FUNC("%s: detect whole chip reset start\n", __func__);
		} else if (rstflag == 2) {	/* Reset end */
			retval = -99;
			BT_INFO_FUNC("%s: detect whole chip reset end\n", __func__);
		}
		goto OUT;
	}

	if (count > 0) {
		if (count < BT_BUFFER_SIZE) {
			write_size = count;
		} else {
			write_size = BT_BUFFER_SIZE;
			BT_ERR_FUNC("%s: count > BT_BUFFER_SIZE\n", __func__);
		}

		if (copy_from_user(&o_buf[0], &buf[0], write_size)) {
			retval = -EFAULT;
			goto OUT;
		}

		written = mtk_wcn_stp_send_data(&o_buf[0], write_size, BT_TASK_INDX);
		if (0 == written) {
			retval = -ENOSPC;
			/* No space is available, native program should not call BT_write with no delay */
			BT_ERR_FUNC
			    ("Packet length %zd, sent length %d, retval = %d\n",
			     count, written, retval);
		} else {
			retval = written;
		}

	} else {
		retval = -EFAULT;
		BT_ERR_FUNC("Packet length %zd is not allowed, retval = %d\n", count, retval);
	}

OUT:
	up(&wr_mtx);
	return retval;
}

ssize_t BT_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
	static int chip_reset_count;
	INT32 retval = 0;
	down(&rd_mtx);

	BT_DBG_FUNC("%s: count %zd pos %lld\n", __func__, count, *f_pos);
	if (rstflag) {
		if (rstflag == 1) {	/* Reset start */
			retval = -88;
			if ((chip_reset_count%500) == 0)
				BT_INFO_FUNC("%s: detect whole chip reset start, %d\n", __func__, chip_reset_count);
			chip_reset_count++;
		} else if (rstflag == 2) {	/* Reset end */
			retval = -99;
			BT_INFO_FUNC("%s: detect whole chip reset end\n", __func__);
			chip_reset_count = 0;
		}
		goto OUT;
	}

	if (count > BT_BUFFER_SIZE) {
		count = BT_BUFFER_SIZE;
		BT_ERR_FUNC("%s: count > BT_BUFFER_SIZE\n", __func__);
	}

	/* Buffer 1 ms for data read gapping*/
	msleep(1);
	retval = mtk_wcn_stp_receive_data(i_buf, count, BT_TASK_INDX);

	while (retval == 0) {	/* Got nothing, wait for STP's signal */
		/*
		* If nonblocking mode, return directly.
		* O_NONBLOCK is specified during open()
		*/
		if (filp->f_flags & O_NONBLOCK) {
			BT_DBG_FUNC("Non-blocking BT_read\n");
			retval = -EAGAIN;
			goto OUT;
		}

		BT_DBG_FUNC("%s: wait_event 1\n", __func__);
		wait_event(BT_wq, flag != 0);
		BT_DBG_FUNC("%s: wait_event 2\n", __func__);
		flag = 0;
		retval = mtk_wcn_stp_receive_data(i_buf, count, BT_TASK_INDX);
		BT_DBG_FUNC("%s: mtk_wcn_stp_receive_data returns %d\n", __func__, retval);
	}

	/* Got something from STP driver */
	if (copy_to_user(buf, i_buf, retval)) {
		retval = -EFAULT;
		goto OUT;
	}

OUT:
	up(&rd_mtx);
	BT_DBG_FUNC("%s: retval = %d\n", __func__, retval);
	return retval;
}

int send_radio_change_cmd(int inst_id,unsigned int enable, unsigned long n1, unsigned long n2)
{
    UINT8 radio_change_cmd[] = {0x01,0xC9,0xFC,0x07,0x01,0x1A,0x0/*inst id*/,0x0,0x0,0x0,0x0};
    UINT8 radio_change_event[7] = {0x4,0xF,0x4,0x0,0x1,0xC9,0xFC};
    UINT8 event_buffer[64] = {0};
    UINT8 ret = 0;
    UINT8 waitcount = 100;
    UINT8 i = 0 ;

    radio_change_cmd[6] = inst_id;
	radio_change_cmd[7] = (n1&0x00FF);
	radio_change_cmd[8] = (n1&0xFF00)>>8;

	radio_change_cmd[9] = (n2&0x00FF);
	radio_change_cmd[10] = (n2&0xFF00)>>8;



	BT_ERR_FUNC("%s print radio_change_cmd %02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n", __func__,
    radio_change_cmd[0],radio_change_cmd[1],radio_change_cmd[2],
    radio_change_cmd[3],radio_change_cmd[4],radio_change_cmd[5],
    radio_change_cmd[6],radio_change_cmd[7],radio_change_cmd[8],
	radio_change_cmd[9],radio_change_cmd[10]);

    ret = mtk_wcn_stp_send_data(radio_change_cmd, sizeof(radio_change_cmd), BT_TASK_INDX);
    if(ret!=sizeof(radio_change_cmd))
        BT_ERR_FUNC("%s send radio_change_cmd return value warning (%d)\n", __func__, ret);

    for(i=0;i<waitcount;i++){
        msleep(100);
        ret = mtk_wcn_stp_receive_data(event_buffer, sizeof(event_buffer), BT_TASK_INDX);
        if(ret==sizeof(radio_change_event)){
            BT_INFO_FUNC("%s receive radio_change_cmd return value warning (%d), waitcount %d, break\n", __func__, ret,i);
            break;
        }
		else
            BT_WARN_FUNC("%s receive radio_change_cmd return value warning (%d), waitcount %d\n", __func__, ret,i);            
    }

    if(ret!=sizeof(radio_change_event))
        BT_ERR_FUNC("%s receive radio_change_event event size error ,return value maybe error (%d)\n", __func__, ret);
		 ret = -EFAULT;
    if(memcmp(event_buffer,radio_change_event,sizeof(radio_change_event))==0){
        BT_INFO_FUNC("%s compare radio_change_event event ok.\n", __func__);
		ret = 0;
	}
    else{
        BT_ERR_FUNC("%s compare radio_change_event event fail\n", __func__);
        ret = -EFAULT;
    }

    BT_INFO_FUNC("%s print event_buffer %02x%02x%02x%02x%02x%02x%02x\n", __func__,
    event_buffer[0],event_buffer[1],event_buffer[2],
    event_buffer[3],event_buffer[4],event_buffer[5],
    event_buffer[6]);

    return ret;
}

#define BT_RADIO_CHANGE_SET_N1     _IOW('H', 30, int)
#define BT_RADIO_CHANGE_SET_N2 	   _IOW('H', 31, int)
#define BT_RADIO_CHANGE_SET_ENABLE _IOW('H', 32, int)
#define BT_RADIO_CHANGE_SET_INSTID _IOW('H', 33, int)

#define BT_LEAVE_FUNC_ON _IOW('w', 1, int)

unsigned long n1 = 0;
unsigned long n2 = 0;
int      inst_id = 0;
/* int BT_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg) */
long BT_unlocked_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	INT32 retval = 0;
	MTK_WCN_BOOL bRet = MTK_WCN_BOOL_TRUE;
	ENUM_WMTHWVER_TYPE_T hw_ver_sym = WMTHWVER_INVALID;

	switch (cmd) {
	case BT_LEAVE_FUNC_ON:
		leave_bt_on = arg;
		BT_DBG_FUNC("# leave_bt_on=%d\n", leave_bt_on);
		break;
	case COMBO_IOC_BT_HWVER:
		/* Get combo HW version */
		hw_ver_sym = mtk_wcn_wmt_hwver_get();
		BT_INFO_FUNC("%s: HW version = %d, sizeof(hw_ver_sym) = %zd\n",
			     __func__, hw_ver_sym, sizeof(hw_ver_sym));
		if (copy_to_user((int __user *)arg, &hw_ver_sym, sizeof(hw_ver_sym)))
			retval = -EFAULT;

		break;

	case COMBO_IOCTL_FW_ASSERT:
		/* Trigger FW assert for debug */
		BT_INFO_FUNC("%s: Host trigger FW assert......, reason:%lu\n", __func__, arg);
		bRet = mtk_wcn_wmt_assert(WMTDRV_TYPE_BT, arg);
		if (bRet == MTK_WCN_BOOL_TRUE) {
			BT_INFO_FUNC("Host trigger FW assert succeed\n");
			retval = 0;
		} else {
			BT_ERR_FUNC("Host trigger FW assert Failed\n");
			retval = (-EBUSY);
		}
		break;
	case COMBO_IOCTL_BT_IC_HW_VER:
		retval = mtk_wcn_wmt_ic_info_get(WMTCHIN_HWVER);
		break;
	case COMBO_IOCTL_BT_IC_FW_VER:
		retval = mtk_wcn_wmt_ic_info_get(WMTCHIN_FWVER);
		break;
    
	case BT_RADIO_CHANGE_SET_N1:
		n1 = arg;
		break;

	case BT_RADIO_CHANGE_SET_N2:
		n2 = arg;
		break;

    case BT_RADIO_CHANGE_SET_INSTID:
        inst_id = arg;
        break;

	case BT_RADIO_CHANGE_SET_ENABLE:
        retval = send_radio_change_cmd(inst_id,1,n1,n2);
		break;

	default:
		retval = -EFAULT;
		BT_ERR_FUNC("Unknown cmd (%d)\n", cmd);
		break;
	}

	return retval;
}

long BT_compat_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	return BT_unlocked_ioctl(filp, cmd, arg);
}

static int BT_open(struct inode *inode, struct file *file)
{
	INT32 n_bts;

	BT_INFO_FUNC("%s: major %d minor %d pid %d\n", __func__, imajor(inode), iminor(inode), current->pid);
	if (current->pid == 1)
		return 0;

	n_bts = atomic_add_return(1, &running_bts);
	if (n_bts == 1)
		BT_DBG_FUNC("# BT func on..\n");
	else {
		BT_DBG_FUNC("# Skip BT func on, n_bts=%d\n", n_bts);
		return 0;
	}

	/* Turn on BT */
	if (MTK_WCN_BOOL_FALSE == mtk_wcn_wmt_func_on(WMTDRV_TYPE_BT)) {
		BT_WARN_FUNC("WMT turn on BT fail!\n");
		return -ENODEV;
	}

	BT_INFO_FUNC("WMT turn on BT OK!\n");
	rstflag = 0;

	if (mtk_wcn_stp_is_ready()) {

		mtk_wcn_stp_set_bluez(0);

		BT_INFO_FUNC("Now it's in MTK Bluetooth Mode\n");
		BT_INFO_FUNC("STP is ready!\n");

		BT_DBG_FUNC("Register BT event callback!\n");
		mtk_wcn_stp_register_event_cb(BT_TASK_INDX, BT_event_cb);
	} else {
		BT_ERR_FUNC("STP is not ready\n");
		mtk_wcn_wmt_func_off(WMTDRV_TYPE_BT);
		return -ENODEV;
	}

	BT_DBG_FUNC("Register BT reset callback!\n");
	mtk_wcn_wmt_msgcb_reg(WMTDRV_TYPE_BT, bt_cdev_rst_cb);

	/* init_MUTEX(&wr_mtx); */
	sema_init(&wr_mtx, 1);
	/* init_MUTEX(&rd_mtx); */
	sema_init(&rd_mtx, 1);
	BT_INFO_FUNC("%s: finish\n", __func__);

	return 0;
}

static int BT_close(struct inode *inode, struct file *file)
{
	INT32 n_bts;

	BT_INFO_FUNC("%s: major %d minor %d pid %d\n", __func__, imajor(inode), iminor(inode), current->pid);
	if (current->pid == 1)
		return 0;

	rstflag = 0;
	mtk_wcn_wmt_msgcb_unreg(WMTDRV_TYPE_BT);
	mtk_wcn_stp_register_event_cb(BT_TASK_INDX, NULL);

	n_bts = atomic_sub_return(1, &running_bts);
	if (n_bts == 0 && !leave_bt_on)
		BT_DBG_FUNC("# BT func off..\n");
	else {
		BT_DBG_FUNC("# Skip BT func off, n_bts=%d leave_bt_on=%d\n",
					n_bts, leave_bt_on);
		return 0;
	}

	if (MTK_WCN_BOOL_FALSE == mtk_wcn_wmt_func_off(WMTDRV_TYPE_BT)) {
		BT_ERR_FUNC("WMT turn off BT fail!\n");
		return -EIO;	/* Mostly, native program will not check this return value. */
	}

	BT_INFO_FUNC("WMT turn off BT OK!\n");
	return 0;
}

const struct file_operations BT_fops = {
	.open = BT_open,
	.release = BT_close,
	.read = BT_read,
	.write = BT_write,
	/* .ioctl = BT_ioctl, */
	.unlocked_ioctl = BT_unlocked_ioctl,
	.compat_ioctl = BT_compat_ioctl,
	.poll = BT_poll
};

static int BT_init(void)
{
	dev_t dev = MKDEV(BT_major, 0);
	INT32 alloc_ret = 0;
	INT32 cdev_err = 0;
#if MTK_BT_HCI && REMOVE_MK_NODE
	INT32 hci_err = 0;
#endif

	/* Static allocate char device */
	alloc_ret = register_chrdev_region(dev, 1, BT_DRIVER_NAME);
	if (alloc_ret) {
		BT_ERR_FUNC("%s: Failed to register char device\n", __func__);
		return alloc_ret;
	}

	cdev_init(&BT_cdev, &BT_fops);
	BT_cdev.owner = THIS_MODULE;

	cdev_err = cdev_add(&BT_cdev, dev, BT_devs);
	if (cdev_err)
		goto error;

#if REMOVE_MK_NODE
	stpbt_class = class_create(THIS_MODULE, "stpbt");
	if (IS_ERR(stpbt_class))
		goto error;
	stpbt_dev = device_create(stpbt_class, NULL, dev, NULL, "stpbt");
	if (IS_ERR(stpbt_dev))
		goto error;
#endif

	BT_INFO_FUNC("%s driver(major %d) installed\n", BT_DRIVER_NAME, BT_major);

	/* Init wait queue */
	init_waitqueue_head(&(inq));

#if MTK_BT_HCI && REMOVE_MK_NODE
	if (!(mtk_hci.hdev = hci_alloc_dev())) {
		BT_ERR_FUNC("%s hci_alloc_dev failed\n", __func__);
		return -ENOMEM;
	}

	mtk_hci.hdev->bus = HCI_SDIO;
	mtk_hci.hdev->open = mtk_bt_hci_open;
	mtk_hci.hdev->close = mtk_bt_hci_close;
	mtk_hci.hdev->send = mtk_bt_hci_send;
	mtk_hci.hdev->flush = mtk_bt_hci_flush;
	mtk_hci.hdev->notify = mtk_bt_hci_notify;
	SET_HCIDEV_DEV(mtk_hci.hdev, stpbt_dev);

	mtk_wcn_stp_register_if_rx(mtk_bt_hci_receive);

	if ((hci_err = hci_register_dev(mtk_hci.hdev))) {
		BT_ERR_FUNC("%s hci_register_dev failed with %d\n", __func__, hci_err);
		hci_free_dev(mtk_hci.hdev);
		return hci_err;
	}

	skb_queue_head_init(&mtk_hci.txq);
	INIT_WORK(&mtk_hci.work, mtk_bt_hci_work);
#endif

	return 0;

error:

#if REMOVE_MK_NODE
	if (!IS_ERR(stpbt_dev))
		device_destroy(stpbt_class, dev);
	if (!IS_ERR(stpbt_class)) {
		class_destroy(stpbt_class);
		stpbt_class = NULL;
	}
#endif
	if (cdev_err == 0)
		cdev_del(&BT_cdev);

	if (alloc_ret == 0)
		unregister_chrdev_region(dev, BT_devs);

	return -1;
}

static void BT_exit(void)
{
	dev_t dev = MKDEV(BT_major, 0);

#if MTK_BT_HCI && REMOVE_MK_NODE
	if (mtk_hci.hdev) {
		hci_unregister_dev(mtk_hci.hdev);
		hci_free_dev(mtk_hci.hdev);
		mtk_hci.hdev = NULL;
	}
#endif

#if REMOVE_MK_NODE
	device_destroy(stpbt_class, dev);
	class_destroy(stpbt_class);
	stpbt_class = NULL;
#endif

	cdev_del(&BT_cdev);
	unregister_chrdev_region(dev, BT_devs);

	BT_INFO_FUNC("%s driver removed\n", BT_DRIVER_NAME);
}

#ifdef MTK_WCN_REMOVE_KERNEL_MODULE

int mtk_wcn_stpbt_drv_init(void)
{
	return BT_init();
}
EXPORT_SYMBOL(mtk_wcn_stpbt_drv_init);

void mtk_wcn_stpbt_drv_exit(void)
{
	return BT_exit();
}
EXPORT_SYMBOL(mtk_wcn_stpbt_drv_exit);

#else

module_init(BT_init);
module_exit(BT_exit);

#endif
