/*
 * Panasonic AVC MMC/SD/SDIO driver
 *
 * Authors: 
 * Copyright (C) 2015-2020 Panasonic Corporation.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

//#define PVCSD_DEBUG
//#define PVCSD_DEBUG_PVCSD_PARAM
#define PVCSD_ENABLE_CH

#include <linux/module.h>
#include <linux/of.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/platform_device.h>
#include <linux/mbus.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/dma-mapping.h>
#include <linux/scatterlist.h>
#include <linux/irq.h>
#include <linux/gpio.h>
#include <linux/mmc/host.h>
#include <linux/mmc/card.h>

#include <asm/sizes.h>
#include <asm/unaligned.h>
#include <asm/cacheflush.h>

#include "mach/pvc.h"
#include "pvc_sdio.h"

#define DRIVER_NAME	"pvsdio"

static int maxfreq = PVCSD_CLOCKRATE_MAX;
static int nodma;
void __iomem *base_local;

#ifdef PVCSD_ENABLE_CH
#define PVCSD_CH_MAX	3
static int ch = -1;
#endif

struct pvcsd_host {
	void __iomem *base;
	void __iomem *a2m_base;
	struct mmc_request *mrq;
	spinlock_t lock;
	unsigned int card_detect;
	unsigned int intr_en1;
	unsigned int intr_en2;
	unsigned int sdio_intr_en1;
	unsigned int sdio_mode;
	unsigned int rw_seq;
	unsigned int option;	
	unsigned int block_size;
	unsigned int pio_size;
	void *pio_ptr;
	unsigned int sg_frags;
	unsigned int* buffer;
	dma_addr_t physical_address;
	unsigned int dma_dir;
	unsigned int total_length;
	unsigned int ns_per_clk;
	unsigned int clock;
	unsigned int base_clock;
	struct timer_list timer;
	struct mmc_host *mmc;
	struct device *dev;
	struct resource *res;
	int irq;
	int irq2;
};

#define pvcsd_write(offs, val)	writel((val), (iobase + (offs)))
#define pvcsd_read(offs)	readl(iobase + (offs)) 

static void pvcsd_dma_set_addr(struct pvcsd_host *host)
{
	void __iomem *iobase;
	u32 tmp;
	iobase = host->base;


	if (host->dma_dir == DMA_TO_DEVICE){
		tmp = (PVCSD_DM_CM_DTRAN_MODE_CH_NUM_DOWN |			/* DMAC channel selector 01 : SD/UHS2 up stream */
			   PVCSD_DM_CM_DTRAN_MODE_BUS_WIDTH_64BIT |	/* Bus width selector 11: 64-bit [default] */
			   PVCSD_DM_CM_DTRAN_MODE_ADDR_MODE);
		pvcsd_write(PVCSD_DM_CM_DTRAN_MODE,tmp);

	}else{
		tmp = (PVCSD_DM_CM_DTRAN_MODE_CH_NUM_UP |			/* DMAC channel selector 01 : SD/UHS2 up stream */
			   PVCSD_DM_CM_DTRAN_MODE_BUS_WIDTH_64BIT |	/* Bus width selector 11: 64-bit [default] */
			   PVCSD_DM_CM_DTRAN_MODE_ADDR_MODE);
		pvcsd_write(PVCSD_DM_CM_DTRAN_MODE,tmp);

	}
	tmp = host->physical_address;
	pvcsd_write(PVCSD_DM_DTRAN_ADDR, tmp);
}


static void pvcsd_dma_start(struct pvcsd_host *host)
{
	void __iomem *iobase;

	iobase = host->base;

#if defined(CONFIG_PVC_PVC02V_BARRIER_REPLACE)
	dsb(st);
#endif
	/* Command type selector 0 : SD Legacy (UHS-I) command									*/
	/* This bit triggers to start up DMAC. this bit to 1, SDIP operates DMAC transfer 		*/
	/* for the channel set by DM_CM_DRAN_MODE.CH_NUM. SDIP clears this bit automatically, 	*/
	/*when SDIP starts up DMAC.[PVCSD_DM_CM_DTRAN_CTRL_DM_START]							*/
	pvcsd_write(PVCSD_DM_CM_DTRAN_CTRL, PVCSD_DM_CM_DTRAN_CTRL_DM_START);
}

static int pvcsd_dma_stop(struct pvcsd_host *host)
{
	void __iomem *iobase;
	u32 ni;
	int nsts;
	u32 tmp;
	iobase = host->base;

	ni = 0;
    nsts = 0;
	if (host->dma_dir == DMA_TO_DEVICE){
		tmp = PVCSD_DM_CM_INFO1_DTRAN_CH_DOWN;	/* Interrupt of DMAC data transfer completion DTRANEND[0]: Channel0 */
	}
	else{
		tmp = PVCSD_DM_CM_INFO1_DTRAN_CH_UP;	/* Interrupt of DMAC data transfer completion DTRANEND[1]: Channel1 */
	}
	while (!(pvcsd_read(PVCSD_DM_CM_INFO1) & tmp))
	{
		if(pvcsd_read(PVCSD_DM_CM_INFO2) & tmp){
			nsts = PVCSD_TRANS_ERROR;
			break;
		}
		if (ni++ > PVCSD_TRANS_WAIT_COUNT){
			nsts = PVCSD_TRANS_ERROR;
			break;
		}
		udelay(1);
	}
#if defined(CONFIG_PVC_PVC02V_BARRIER_REPLACE)
	dmb();
#endif
	if(nsts == PVCSD_TRANS_ERROR){
		pvcsd_write(PVCSD_DM_CM_INFO2,~(tmp));
	}
	pvcsd_write(PVCSD_DM_CM_INFO1,~(tmp));
	return nsts;
}

static int pvcsd_write_protect(struct pvcsd_host *host)
{
    void __iomem *iobase = host->base;

	int ret = 0;
	if (!(pvcsd_read(PVCSD_INFO1) & PVCSD_INFO1_WRITE_PROTECT)) {
		ret = 1;
	} else {
		ret = 0;
	}
	return ret;
}

static void pvcsd_clock(struct pvcsd_host *host, unsigned int clk)
{
	void __iomem *iobase = host->base;
	unsigned long flags;
	u32 tmp;
	if (!(pvcsd_read(PVCSD_INFO2) & PVCSD_STATE_CBSY)) {
		spin_lock_irqsave(&host->lock, flags);
		tmp = pvcsd_read(PVCSD_CLK_CTRL);
		tmp &= ~PVCSD_SD_ENABLE_CLK;
		pvcsd_write(PVCSD_CLK_CTRL,tmp);
		switch (clk) {
			case PVCSD_CLKDEV_DATA:
			tmp |= PVCSD_CLKDEV_DIV2;
			break;

			case PVCSD_CLKDEV_IDENT:
			case PVCSD_CLKDEV_STBY:
			tmp |= PVCSD_CLKDEV_DIV128;
			break;

			default:
			tmp |= PVCSD_CLKDEV_DIV128;
			break;
		}
		pvcsd_write(PVCSD_CLK_CTRL,tmp);
		if (clk){
			tmp |= PVCSD_SD_ENABLE_CLK;
			pvcsd_write(PVCSD_CLK_CTRL,tmp);
		}
		spin_unlock_irqrestore(&host->lock, flags);
	}
	return;
}

static int pvcsd_setup_data(struct pvcsd_host *host, struct mmc_data *data)
{
	void __iomem *iobase = host->base;

	pvcsd_write(PVCSD_INFO1,0);
	pvcsd_write(PVCSD_INFO2,0);
	pvcsd_write(PVCSD_EXT_CD,0);
	pvcsd_write(PVCSD_EXT_CD_DAT3,0);

	host->option &= ~PVCSD_HOST_CTRL_TMOUT_MASK;
	host->option |= PVCSD_HOST_CTRL_TMOUT_EN;
	host->block_size = data->blksz;

	pvcsd_write(PVCSD_OPTION, host->option);

	if( (data->blksz * data->blocks) % 8 ){
		nodma = 1;
	}else{
		nodma = 0;
	}
    pvcsd_clock(host,PVCSD_CLKDEV_DATA);
	pvcsd_write(PVCSD_SIZE, data->blksz);
	pvcsd_write(PVCSD_INFO1_MASK, PVCSD_INFO1_INSERT_DEF);
	pvcsd_write(PVCSD_INFO2_MASK, (PVCSD_RX_READY | PVCSD_TX_AVAIL));
	pvcsd_write(PVCSD_STOP, PVCSD_AUTO_STOP);
	pvcsd_write(PVCSD_SECCNT, data->blocks);

	if (nodma) {
		pvcsd_write(PVCSD_CC_EXT_MODE,0);

		/* 16bit */
		pvcsd_write(PVCSD_HOST_MODE,0x0101);		/* 1: Buffer access by 16-bit(BUSWIDTH=0) / 32-bit(BUSWIDTH=1) unit mode(SD_BUF [15:0]) 	*/

		host->pio_size = data->blocks * data->blksz;
		host->pio_ptr = sg_virt(data->sg);
		if (!nodma)
			printk(KERN_DEBUG "%s: fallback to PIO for data "
					  "at 0x%p size %d\n",
					  mmc_hostname(host->mmc),
					  host->pio_ptr, host->pio_size);
		return 1;
	} else {

		pvcsd_write(PVCSD_CC_EXT_MODE,PVCSD_DMA_MODE);

		/* 64bit */
		pvcsd_write(PVCSD_HOST_MODE, 0);

		host->dma_dir = (data->flags & MMC_DATA_READ) ?
			DMA_FROM_DEVICE : DMA_TO_DEVICE;



		host->sg_frags = dma_map_sg(mmc_dev(host->mmc), data->sg,
					    data->sg_len, host->dma_dir);
		host->physical_address = sg_dma_address(data->sg);
		host->total_length = data->blocks * data->blksz;
		pvcsd_dma_set_addr(host);

		return 0;
	}
}

static u32 conv_cmd(u32 cmd)
{
    u32 command_set;
    switch (cmd) {
    case ACMD6:
    case ACMD13:
    case ACMD22:
    case ACMD23:
    case ACMD41:
    case ACMD42:
    case ACMD51:
        command_set = (short)cmd;
        command_set |= (1 << 6);
        command_set &= ~(1 << 7);
        command_set &= ~((1 << 10) | (1 << 9) | (1 << 8));
        break;
    
    default:
        command_set = (short)cmd;
        command_set &= ~((1 << 6) | (1 << 7));
        command_set &= ~((1 << 10) | (1 << 9) | (1 << 8)); 
        break;
    }
    return command_set;
}

static void pvcsd_request(struct mmc_host *mmc, struct mmc_request *mrq)
{
	struct pvcsd_host *host = mmc_priv(mmc);
	void __iomem *iobase = host->base;
	struct mmc_command *cmd = mrq->cmd;
	u32 cmdreg = 0, intr_info1 = 0, intr_info2 = 0;
	unsigned long flags;

	BUG_ON(host->mrq != NULL);
	host->mrq = mrq;
	host->rw_seq = 0;
	dev_dbg(host->dev, "cmd %d (hw state 0x%04x)\n",
		cmd->opcode, pvcsd_read(PVCSD_ERR_STS1));
	cmdreg = conv_cmd(cmd->opcode);

	if ((cmd->opcode == ACMD6) || (cmd->opcode == ACMD8)){
		if ((cmd->opcode == CMD6) && (mrq->data)){
		        cmdreg &= ~(1 << 6);
			host->rw_seq = 1;
		}
		if (cmd->flags & MMC_RSP_BUSY)
			cmdreg |= PVCSD_CMD_RSP_48BUSY;
		else if (cmd->flags & MMC_RSP_136)
			cmdreg |= PVCSD_CMD_RSP_136;
		else if (cmd->flags & MMC_RSP_PRESENT)
			cmdreg |= PVCSD_CMD_RSP_48;
		else
			cmdreg |= PVCSD_CMD_RSP_NONE;
	}
	if ((cmd->opcode == CMD17) || (cmd->opcode == CMD24) ||
	    (cmd->opcode == CMD18) || (cmd->opcode == CMD25) ||
            (cmd->opcode == ACMD51) || (cmd->opcode == CMD53)) {
		host->rw_seq = 1;
		if ((cmd->opcode == CMD18) || (cmd->opcode == CMD25)) {
			cmdreg |= (1 << 13);
		}
	}else{
		pvcsd_write(PVCSD_STOP, 0);
		pvcsd_write(PVCSD_SECCNT, 0);
		pvcsd_write(PVCSD_INFO1_MASK, PVCSD_INFO1_INSERT_DEF);
		pvcsd_write(PVCSD_INFO2_MASK, PVCSD_INFO2_RESET_MASK);
	}

	if (((cmd->opcode == CMD5) || (cmd->opcode == CMD52) ||
		(cmd->opcode == CMD53) ||(cmd->opcode == CMD54)) ||
		(mmc->card && mmc->card->type == MMC_TYPE_SDIO)){
		    cmdreg &= ~((1 << 6) | (1 << 7));
		if (cmd->opcode == CMD5){
		    cmdreg |= ((1 << 10) | (1 << 9) | (1 << 8));
		}else{
		    cmdreg &= ~((1 << 9) | (1 << 8));
		    cmdreg |= (1 << 10);
		}
	}
	if (cmd->opcode == CMD53)
		    cmdreg |= (1 << 14);


	if (mrq->data) {
		struct mmc_data *data = mrq->data;
		int pio;
		cmdreg |= PVCSD_CMD_DATA_PRESENT;
		if (data->flags & MMC_DATA_READ){
			cmdreg |= PVCSD_CMD_READ;
			intr_info2 |= PVCSD_NOR_BRE;
		}else{
			intr_info2 |= PVCSD_NOR_BWE;
		}
		intr_info1 &= ~PVCSD_NOR_RW_DONE;
		pio = pvcsd_setup_data(host, data);
		intr_info1 |= PVCSD_NOR_RW_DONE;
		if ((data->blocks > 1) && (cmd->opcode == CMD53)){
			cmdreg |= (1 << 13);
		}
	} else {
		if (cmd->opcode == CMD13) cmdreg &= ~((1 << 6) | (1 << 7));
		intr_info1 |= PVCSD_NOR_CMD_DONE;
	}

	pvcsd_write(PVCSD_ARG0, cmd->arg & 0xffff);
	pvcsd_write(PVCSD_ARG1,  cmd->arg >> 16);

#ifdef PVCSD_DEBUG /* For ES test */
	printk("\nSD CMD Send: %d\n",cmd->opcode);
	printk("ARG0: %X ARG1: %X\n",pvcsd_read(PVCSD_ARG0),pvcsd_read(PVCSD_ARG1));
#endif
	spin_lock_irqsave(&host->lock, flags);

	pvcsd_write(PVCSD_SDIO_INFO1, ~PVCSD_CARD_INT);
	pvcsd_write(PVCSD_SDIO_INFO1_MASK, 0x0002);

	host->sdio_intr_en1 &= PVCSD_CARD_INT;
	host->intr_en1 |= intr_info1;
	host->intr_en2 |= intr_info2;
	pvcsd_write(PVCSD_INFO1, ~(host->intr_en1));
	pvcsd_write(PVCSD_INFO1_MASK, 
		pvcsd_read(PVCSD_INFO1_MASK) & ~(host->intr_en1));
	pvcsd_write(PVCSD_INFO2, ~(host->intr_en2));

	mod_timer(&host->timer, jiffies + 5 * HZ);

	pvcsd_write(PVCSD_CMD, cmdreg);

	spin_unlock_irqrestore(&host->lock, flags);
}

static u32 pvcsd_finish_cmd(struct pvcsd_host *host, struct mmc_command *cmd,
			   u32 err_status)
{
	void __iomem *iobase = host->base;
#ifdef PVCSD_DEBUG /* For ES test */
	unsigned int j;
#endif

	if (cmd->flags & MMC_RSP_136) {
		unsigned int response[8], i;
		for (i = 0; i < 8; i++)
			response[i] = pvcsd_read(PVCSD_RSP(i));
		cmd->resp[0] =		((response[7] & 0xff) << 24) |
					((response[6] & 0xffff) << 8) |
					((response[5] & 0xff00) >> 8);
		cmd->resp[1] =		((response[5] & 0xff) << 24) |
					((response[4] & 0xffff) << 8) |
					((response[3] & 0xff00) >> 8);
		cmd->resp[2] =		((response[3] & 0xff) << 24) |
					((response[2] & 0xffff) << 8) |
					((response[1] & 0xff00) >> 8);
		cmd->resp[3] =		((response[1] & 0xff) << 24) |
					((response[0] & 0xffff) << 8);
	} else if (cmd->flags & MMC_RSP_PRESENT) {
		unsigned int response[3];
		response[0] = pvcsd_read(PVCSD_RSP(0));
		response[1] = pvcsd_read(PVCSD_RSP(1));
		response[2] = pvcsd_read(PVCSD_RSP(2));
		cmd->resp[0] = (response[0] & 0xffff) | 
			       ((response[1] & 0xffff) << 16);
		cmd->resp[1] = (response[2] & 0x3f);
		cmd->resp[2] = 0;
		cmd->resp[3] = 0;
	}

#ifdef PVCSD_DEBUG /* For ES test */
	for (j = 0; j < 8; j++){
		printk("RSP%d: %X = %X\n",j,PVCSD_RSP(j),pvcsd_read(PVCSD_RSP(j)));
	}
#endif

	if (err_status & PVCSD_ERR_RSP_TIMEOUT) {
		cmd->error = -ETIMEDOUT;
	} else if (err_status & (PVCSD_ERR_CRC | PVCSD_ERR_END |
				 PVCSD_ERR_CMD )) {
		cmd->error = -EILSEQ;
	}

	err_status &= ~(PVCSD_ERR_RSP_TIMEOUT | PVCSD_ERR_CRC |
			PVCSD_ERR_END | PVCSD_ERR_CMD );

	host->rw_seq = 0;
	return err_status;
}

static u32 pvcsd_finish_data(struct pvcsd_host *host, struct mmc_data *data,
			    u32 err_status)
{
	void __iomem *iobase = host->base;

	if (host->pio_ptr) {
		host->pio_ptr = NULL;
		host->pio_size = 0;
		/* BWE/BRE mask */
		pvcsd_write(PVCSD_INFO2,~(PVCSD_RX_READY | PVCSD_TX_AVAIL));
		pvcsd_write(PVCSD_INFO2_MASK,PVCSD_INFO2_RESET_MASK);
	} else {
		dma_unmap_sg(mmc_dev(host->mmc), data->sg, host->sg_frags,
			     (data->flags & MMC_DATA_READ) ?
				DMA_FROM_DEVICE : DMA_TO_DEVICE);
	}

	if (err_status & PVCSD_ERR_DATA_TIMEOUT)
		data->error = -ETIMEDOUT;
	else if (err_status & (PVCSD_ERR_CRC | PVCSD_ERR_END))
		data->error = -EILSEQ;

	err_status &= ~(PVCSD_ERR_DATA_TIMEOUT | PVCSD_ERR_CRC |
			PVCSD_ERR_END);

	data->bytes_xfered = data->blocks * data->blksz;
	/* (data->blocks - pvcsd_read(PVCSD_CURR_BLK_LEFT)) * data->blksz;*/

	/* We can't be sure about the last block when errors are detected */
	if (data->bytes_xfered && data->error)
		data->bytes_xfered -= data->blksz;

	if (data->stop) {

		unsigned int response[3];
		response[0] = pvcsd_read(PVCSD_AUTO_RSP(0));
		response[1] = pvcsd_read(PVCSD_AUTO_RSP(1));
		response[2] = pvcsd_read(PVCSD_AUTO_RSP(2));
		data->stop->resp[0] = (response[0] & 0xffff) | 
			       ((response[1] & 0xffff) << 16);
		data->stop->resp[1] = (response[2] & 0x3f);
		data->stop->resp[2] = 0;
		data->stop->resp[3] = 0;

		if (err_status & PVCSD_ERR_AUTOCMD12) {
			dev_dbg(host->dev, "c12err 0x%04x\n", err_status);
			err_status &= ~PVCSD_ERR_AUTOCMD12;
		}
	}

       	pvcsd_write(PVCSD_INFO1_MASK,
		pvcsd_read(PVCSD_INFO1_MASK) | PVCSD_NOR_RW_DONE);
	return err_status;
}

static irqreturn_t pvcsd_irq(int irq, void *dev)
{
	struct pvcsd_host *host = dev;
	void __iomem *iobase = host->base;
	unsigned long flags;
	u32 intr_info1, intr_info2, sdio_info1;
	int irq_handled = 0;

#ifdef PVCSD_DEBUG /* For ES test */
	printk("### IRQ ### \n");
	printk("INFO1: %X INFO2: %X SDIO_INFO1: %X \n",pvcsd_read(PVCSD_INFO1),pvcsd_read(PVCSD_INFO2),pvcsd_read(PVCSD_SDIO_INFO1));
	printk("ERR_STS1: %X ERR_STS2: %X \n",pvcsd_read(PVCSD_ERR_STS1),pvcsd_read(PVCSD_ERR_STS2));
	printk("EXT_CD: %X EXT_CD_DAT3: %X EXT_SDIO: %X \n",pvcsd_read(PVCSD_EXT_CD),pvcsd_read(PVCSD_EXT_CD_DAT3),pvcsd_read(PVCSD_EXT_SDIO));
	printk("INFO1_MASK: %X INFO2_MASK: %X SDIO_INFO1_MASK: %X \n",pvcsd_read(PVCSD_INFO1_MASK),pvcsd_read(PVCSD_INFO2_MASK),pvcsd_read(PVCSD_SDIO_INFO1_MASK));
	printk("EXT_CD_MASK: %X  EXT_CD_DAT3_MASK: %X \n",pvcsd_read(PVCSD_EXT_CD_MASK),pvcsd_read(PVCSD_EXT_CD_DAT3_MASK));
//	printk("DMA STATUS %X \n" ,pvcsd_read(PVCSD_SDTOP_DMA_STATUS));
	printk("########### \n");
#endif

	if ((host->rw_seq > 1) && (!host->pio_size) && (nodma == 0)){
		pvcsd_dma_stop(host);
	}
	spin_lock_irqsave(&host->lock, flags);
	intr_info1 = pvcsd_read(PVCSD_INFO1) & ~(PVCSD_INFO1_SDDAT3_DETECT);
	intr_info2 = pvcsd_read(PVCSD_INFO2) & ~(PVCSD_INFO2_DAT0);
	sdio_info1 = pvcsd_read(PVCSD_SDIO_INFO1);

	/* v`FbN(XvAXG[΍) */
	if( !(intr_info1 & 0x0000003f) && !(intr_info2 & 0x0000C37F) && !(sdio_info1 & 0x00000001) ){
		/* ׂv͗ĂȂreturn */
		spin_unlock_irqrestore(&host->lock, flags);
		return IRQ_HANDLED;
	}

	dev_dbg(host->dev, "INFO1=0x%04x, INFO2=0x%04x "
			   "ERR_STS1=0x%04x ERR_STS2=0x%04x "
			   "INFO1_MASK=0x%04x INFO2_MASK=0x%04x \n", 
				intr_info1,intr_info2,
				pvcsd_read(PVCSD_ERR_STS1),
				pvcsd_read(PVCSD_ERR_STS2),
				pvcsd_read(PVCSD_INFO1_MASK),
				pvcsd_read(PVCSD_INFO2_MASK));

	/* Card Detection:Insertion */
	if (intr_info1 & PVCSD_INFO1_CARD_INSERT &&
		 !(intr_info1 & PVCSD_INFO1_RESPONSE_END)) {

		intr_info1 = (PVCSD_INFO1_INSERT_DEF |
				PVCSD_INFO1_RESPONSE_END);

		pvcsd_write(PVCSD_INFO1,~(PVCSD_INFO1_CARD_INSERT |
				PVCSD_INFO1_CARD_DETECT));
     		pvcsd_write(PVCSD_INFO1_MASK,intr_info1);
		host->intr_en1 = ~intr_info1;

		mmc_detect_change(host->mmc, msecs_to_jiffies(100));
		spin_unlock_irqrestore(&host->lock, flags);
		return IRQ_HANDLED;
	}

    	/* Card Detection:Removal */
	if (intr_info1 & PVCSD_INFO1_CARD_REMOVE) {

		intr_info1 = (PVCSD_INFO1_REMOVE_DEF |
				PVCSD_INFO1_RESPONSE_END);

		pvcsd_write(PVCSD_INFO1,~(PVCSD_INFO1_CARD_REMOVE |
				PVCSD_INFO1_CARD_DETECT));

       		pvcsd_write(PVCSD_INFO1_MASK,intr_info1);
		host->intr_en1 = ~intr_info1;
		mmc_detect_change(host->mmc, msecs_to_jiffies(100));
		spin_unlock_irqrestore(&host->lock, flags);
		return IRQ_HANDLED;
	}
	if (host->rw_seq == 1 && (intr_info1 & PVCSD_INFO1_RESPONSE_END)){
		struct mmc_request *mrq = host->mrq;
		struct mmc_data *data = mrq->data;
		pvcsd_write(PVCSD_INFO1, ~(PVCSD_INFO1_RESPONSE_END));
		pvcsd_read(PVCSD_RSP(0));
		pvcsd_read(PVCSD_RSP(1));
		if (host->pio_size){
			/* BWE/BRE Unmask */
			if(data->flags & MMC_DATA_READ){
				/* BRE Unmask(BWE mask) */
				pvcsd_write(PVCSD_INFO2_MASK,PVCSD_TX_AVAIL);
			}else{
				/* BWE Unmask(BRE mask) */
				pvcsd_write(PVCSD_INFO2_MASK,PVCSD_RX_READY);
			}
		}
		del_timer(&host->timer);

		host->rw_seq++;
		if (!host->pio_size) pvcsd_dma_start(host);
		spin_unlock_irqrestore(&host->lock, flags);
		return IRQ_HANDLED;
	}

	if (!(intr_info1 & PVCSD_NOR_RW_DONE) && host->rw_seq > 1 && !sdio_info1){
		if (host->pio_size &&
		    (intr_info2 & host->intr_en2 &
		     PVCSD_RX_READY)) {
			int len = 0;	
			u32 *p = (u32 *)host->pio_ptr;
			int s = host->block_size;

			if (intr_info2 & PVCSD_RX_READY) {
				pvcsd_write(PVCSD_INFO2, ~(intr_info2));
			}

			while (len < s) {
				*p = (u32)(pvcsd_read(PVCSD_BUF0) & 0xffffffff);
				len += 4;
				p++;
			}
			host->pio_ptr = p;
			host->pio_size -= s;

			spin_unlock_irqrestore(&host->lock, flags);
			return IRQ_HANDLED;
		} else if (host->pio_size &&
			   (intr_info2 & host->intr_en2 &
			    PVCSD_TX_AVAIL)) {
			int len = 0;	
			u32 *p = host->pio_ptr;
			int s = host->block_size;

			if (intr_info2 & PVCSD_TX_AVAIL) {
				pvcsd_write(PVCSD_INFO2, ~(intr_info2));
			}

			while (len < s) {
				pvcsd_write(PVCSD_BUF0,*p++);
				len += 4;
			}
			dev_dbg(host->dev, "pio %d intr 0x%04x err_sts1 0x%04x"
					" err_sts2 0x%04x\n",s, intr_info1, 
					pvcsd_read(PVCSD_ERR_STS1),
					pvcsd_read(PVCSD_ERR_STS2));
			host->pio_ptr = p;
			host->pio_size -= s;
			irq_handled = 1;
			spin_unlock_irqrestore(&host->lock, flags);
			return IRQ_HANDLED;
		}
	}

	pvcsd_write(PVCSD_INFO1, ~(intr_info1));
	if(nodma){
		/* PIO mode SD_INFO2 mask(except BWE/RWE) */
		pvcsd_write(PVCSD_INFO2, (PVCSD_RX_READY | PVCSD_TX_AVAIL));
	}else{
		/* DMA mode SD_INFO2 mask(All clear) */
		pvcsd_write(PVCSD_INFO2, ~(intr_info2));
	}
	pvcsd_write(PVCSD_SDIO_INFO1,~(sdio_info1));

	if ((sdio_info1 & host->sdio_intr_en1 & ~(PVCSD_CARD_INT)) ||
            (intr_info1 & (PVCSD_NOR_CMD_DONE | PVCSD_NOR_RW_DONE))) {

		struct mmc_request *mrq = host->mrq;
		struct mmc_command *cmd = mrq->cmd;
		u32 err_status = 0;

		del_timer(&host->timer);

		host->mrq = NULL;

		host->sdio_intr_en1 &= PVCSD_CARD_INT;
		pvcsd_write(PVCSD_SDIO_INFO1, ~host->sdio_intr_en1);

		if (intr_info2 & PVCSD_ERROR_BIT) {
			dev_dbg(host->dev, "err 0x%04x\n",
				(intr_info2 & PVCSD_ERROR_BIT));
			err_status = intr_info2;
		}

		err_status = pvcsd_finish_cmd(host, cmd, err_status);
		if (mrq->data)
			err_status = pvcsd_finish_data(host, mrq->data, err_status);
		if (err_status) {
			printk(KERN_ERR "%s: unhandled error status %#04x\n",
					mmc_hostname(host->mmc), err_status);
			cmd->error = -ENOMSG;
		}
	    	if (intr_info1 & PVCSD_INFO1_CARD_DETECT) {
		       	pvcsd_write(PVCSD_INFO1_MASK,PVCSD_INFO1_INSERT_DEF);
		}else{
	       		pvcsd_write(PVCSD_INFO1_MASK,PVCSD_INFO1_REMOVE_DEF);
		}
		host->rw_seq = 0;
		mmc_request_done(host->mmc, mrq);
		irq_handled = 1;

	}

	if (sdio_info1 & PVCSD_IO_IRQ) {
		/*
		 * avoid deadlock:
		 * mmc_signal_sdio_irq calls enable_sdio_irq()
		 */
		spin_unlock_irqrestore(&host->lock,flags);
		mmc_signal_sdio_irq(host->mmc);
		return IRQ_HANDLED;
	}

	pvcsd_write(PVCSD_SDIO_INFO1, ~host->sdio_intr_en1);

	if (irq_handled) {
		spin_unlock_irqrestore(&host->lock, flags);
		return IRQ_HANDLED;
	}

	printk(KERN_ERR "%s: unhandled interrupt status1=0x%04x status2=0x%04x"
			" en1=0x%04x en2=0x%04x "
			"pio=%d\n", mmc_hostname(host->mmc), intr_info1,
			intr_info2,
			host->intr_en1,host->intr_en2, host->pio_size);
	spin_unlock_irqrestore(&host->lock, flags);
	return IRQ_HANDLED;
}

static void pvcsd_timeout_timer(unsigned long data)
{
	struct pvcsd_host *host = (struct pvcsd_host *)data;
	void __iomem *iobase = host->base;
	struct mmc_request *mrq;
	unsigned long flags;

	spin_lock_irqsave(&host->lock, flags);
	mrq = host->mrq;
	if (mrq) {
		printk(KERN_ERR "%s: Timeout waiting for hardware interrupt.\n",				mmc_hostname(host->mmc));

#ifdef PVCSD_DEBUG /* For ES test */
		printk("INFO1: %X  INFO2: %X\n",pvcsd_read(PVCSD_INFO1),pvcsd_read(PVCSD_INFO2));
		printk("INFO1_MASK: %X  INFO2_MASK: %X SDIO_INFO1_MASK: %X \n",pvcsd_read(PVCSD_INFO1_MASK),pvcsd_read(PVCSD_INFO2_MASK),pvcsd_read(PVCSD_SDIO_INFO1_MASK));
#endif

		host->mrq = NULL;

		pvcsd_write(PVCSD_SOFT_RST, 0);
		udelay(1);
		pvcsd_write(PVCSD_SOFT_RST, 7);

		pvcsd_write(PVCSD_INFO1, 0);
		pvcsd_write(PVCSD_INFO2, 0);
		pvcsd_write(PVCSD_SDIO_INFO1, 0);
		pvcsd_write(PVCSD_SDIO_INFO1_MASK, 0);

		mrq->cmd->error = -ETIMEDOUT;
		pvcsd_finish_cmd(host, mrq->cmd, 0);
		if (mrq->data) {
			mrq->data->error = -ETIMEDOUT;
			pvcsd_finish_data(host, mrq->data, 0);
		}
	}
	spin_unlock_irqrestore(&host->lock, flags);

	if (mrq)
		mmc_request_done(host->mmc, mrq);
}

static void pvcsd_enable_sdio_irq(struct mmc_host *mmc, int enable)
{
	struct pvcsd_host *host = mmc_priv(mmc);
	void __iomem *iobase = host->base;
	unsigned long flags;

	spin_lock_irqsave(&host->lock, flags);
	if (enable) {
		host->sdio_intr_en1 &= ~PVCSD_CARD_INT;
		host->sdio_mode |= PVCSD_SDIO_ENABLE;
	} else {
		host->sdio_intr_en1 |= PVCSD_CARD_INT;
		host->sdio_mode &= ~PVCSD_SDIO_ENABLE;
	}
	pvcsd_write(PVCSD_SDIO_INFO1, 0);
	pvcsd_write(PVCSD_SDIO_INFO1_MASK, ~host->sdio_intr_en1);
	pvcsd_write(PVCSD_SDIO_MODE, host->sdio_mode);
	spin_unlock_irqrestore(&host->lock, flags);
}

static int pvcsd_get_ro(struct mmc_host *mmc)
{
	struct pvcsd_host *host = mmc_priv(mmc);

	return pvcsd_write_protect(host);
}

static void pvcsd_reset(struct pvcsd_host *host)
{
	void __iomem *iobase = host->base;
	pvcsd_write(PVCSD_OPTION, host->option);
}

static void pvcsd_power_up(struct pvcsd_host *host)
{
	void __iomem *iobase = host->base;
	dev_dbg(host->dev, "power up\n");
	
	pvcsd_write(PVCSD_INFO1, 0);
	pvcsd_write(PVCSD_INFO2, 0);
	pvcsd_write(PVCSD_SDIO_INFO1, 0);
	pvcsd_write(PVCSD_SDIO_INFO1_MASK, 0);
}

static void pvcsd_power_down(struct pvcsd_host *host)
{
	dev_dbg(host->dev, "power down\n");
}

static void pvcsd_set_ios(struct mmc_host *mmc, struct mmc_ios *ios)
{
	struct pvcsd_host *host = mmc_priv(mmc);
	void __iomem *iobase = host->base;
	u32 option_reg = 0;
	u32 div;

	if (ios->power_mode == MMC_POWER_UP){
		pvcsd_power_up(host);
	}

	if (ios->clock == 0) {
		pvcsd_clock(host, PVCSD_CLKDEV_STOP);
		host->clock = 0;
		dev_dbg(host->dev, "clock off\n");
	} else if (ios->clock != host->clock) {

		for (div = 2;div <= 1024;div *= 2) {
			if ((host->base_clock / div) <= ios->clock){
				break;
			}
		}
		div >>= 2;
		div |= PVCSD_SD_ENABLE_CLK;
		pvcsd_write(PVCSD_CLK_CTRL, div);

		host->clock = ios->clock;
		host->ns_per_clk = 1000000000 / (host->base_clock / (div << 1));		dev_dbg(host->dev, "clock=%d (%d), div=0x%04x\n",
			ios->clock, host->base_clock / (div << 1), (div << 1));
	}

	option_reg |= PVCSD_HOST_CTRL_TMOUT_EN;

	if (ios->bus_width == MMC_BUS_WIDTH_4)
		option_reg &= ~PVCSD_HOST_CTRL_DATA_WIDTH_4_BITS;
	else
		option_reg |= PVCSD_HOST_CTRL_DATA_WIDTH_4_BITS;
	host->option = option_reg;
	pvcsd_write(PVCSD_OPTION, host->option);
	dev_dbg(host->dev, "option 0x%04x: %s\n", option_reg,
		(option_reg & PVCSD_HOST_CTRL_DATA_WIDTH_4_BITS) ?
			"1bit-width" : "4bit-width");

	if (ios->power_mode == MMC_POWER_OFF)
		pvcsd_power_down(host);
}

static void pvcsd_init_card(struct mmc_host *host, struct mmc_card *card)
{
	if (card) {
		/* Set a flag to Card Detecton Disable(1) */
		card->quirks |= MMC_QUIRK_DISABLE_CD;
	}
}

static const struct mmc_host_ops pvcsd_ops = {
	.request		= pvcsd_request,
	.get_ro			= pvcsd_get_ro,
	.set_ios		= pvcsd_set_ios,
	.enable_sdio_irq	= pvcsd_enable_sdio_irq,
	.init_card		= pvcsd_init_card,
};

static int __init pvcsd_probe(struct platform_device *pdev)
{
	struct mmc_host *mmc = NULL;
	struct pvcsd_host *host = NULL;
#ifdef CONFIG_OF
	struct device_node *np = pdev->dev.of_node;
#else
	const struct pvcsdio_platform_data *pvcsd_data;
#endif
	struct resource *r;
	int ret, irq,irq2;

	void __iomem *iobase;

#ifdef PVCSD_ENABLE_CH
	static int probe_cnt = 0;

	if (probe_cnt != ch) {
		probe_cnt++;
		return 0;
	}
#endif

//	nodma = 1; // FORCE PIO

	r = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	irq = platform_get_irq(pdev, 0);
	irq2 = platform_get_irq(pdev, 1);

#ifdef PVCSD_DEBUG_PVCSD_PARAM
	printk("%s: r %#x r->start %#x\n", __func__, (int)r, r?(int)(r->start):0);
	printk("%s: irq %d\n", __func__, irq);
	printk("%s: irq2 %d\n", __func__, irq2);
#endif /* PVCSD_DEBUG_PVCSD_PARAM */
#ifdef PVCSD_DEBUG /* For ES test */
	printk("Request IRQ No. %d\n",irq);
	printk("Request IRQ2 No. %d\n",irq2);
#endif
#ifdef CONFIG_OF
	if (!r || irq < 0 || irq2 < 0 || !np)
		return -ENXIO;
#else
	pvcsd_data = pdev->dev.platform_data;
	if (!r || irq < 0 || irq2 < 0 || !pvcsd_data)
		return -ENXIO;
#endif

	r = request_mem_region(r->start, SZ_1K, DRIVER_NAME);
	if (!r)
		return -EBUSY;

	mmc = mmc_alloc_host(sizeof(struct pvcsd_host), &pdev->dev);
	if (!mmc) {
		ret = -ENOMEM;
		goto out;
	}

	host = mmc_priv(mmc);
	host->mmc = mmc;
	host->dev = &pdev->dev;
	host->res = r;
#ifdef CONFIG_OF
	of_property_read_u32(np, "max-frequency", &host->base_clock);
#else
	host->base_clock = pvcsd_data->clock;
#endif
	mmc->ops = &pvcsd_ops;
	mmc->ocr_avail = MMC_VDD_32_33 | MMC_VDD_33_34;
	mmc->caps = MMC_CAP_4_BIT_DATA | MMC_CAP_SDIO_IRQ |
		    MMC_CAP_SD_HIGHSPEED | MMC_CAP_MMC_HIGHSPEED;
	mmc->f_min = maxfreq >> 16;
	mmc->f_max = maxfreq;

	mmc->max_blk_size = 2048;
	mmc->max_blk_count = 65535;

	mmc->max_segs = 1;
	mmc->max_seg_size = mmc->max_blk_size * mmc->max_blk_count;
	mmc->max_req_size = mmc->max_blk_size * mmc->max_blk_count;

	spin_lock_init(&host->lock);

#ifdef PVCSD_DEBUG /* For ES test */
	printk("SD_TOP REG BaseAddr  %X\n",r->start);
#endif
	host->base = ioremap(r->start, SZ_4K);

	base_local = host->base;
	if (!host->base) {
		ret = -ENOMEM;
		goto out;
	}

	pvcsd_reset(host);
	pvcsd_power_down(host);

	iobase = host->base;
	pvcsd_write(PVCSD_SOFT_RST, 7);
	pvcsd_write(PVCSD_OPTION, PVCSD_HOST_CTRL_TMOUT_EN);
	pvcsd_write(PVCSD_INFO1_MASK, PVCSD_INFO1_STMASK);
	pvcsd_write(PVCSD_INFO2_MASK, 
		(pvcsd_read(PVCSD_INFO2_MASK) | PVCSD_INFO2_DAT0));
	pvcsd_clock(host,PVCSD_CLKDEV_IDENT);

	ret = request_irq(irq, pvcsd_irq, 0, DRIVER_NAME, host);
	if (ret) {
		printk(KERN_ERR "%s: cannot assign irq %d\n", DRIVER_NAME, irq);
		goto out;
	} else
		host->irq = irq;

	ret = request_irq(irq2, pvcsd_irq, 0, DRIVER_NAME, host);
	if (ret) {
		printk(KERN_ERR "%s: cannot assign irq %d\n", DRIVER_NAME, irq2);
		goto out;
	} else
		host->irq2 = irq2;


	setup_timer(&host->timer, pvcsd_timeout_timer, (unsigned long)host);
	platform_set_drvdata(pdev, mmc);
	ret = mmc_add_host(mmc);
	if (ret)
		goto out;

	printk(KERN_NOTICE "%s: %s driver initialized,\n ",
			   mmc_hostname(mmc), DRIVER_NAME);
	return 0;

out:
	if (host) {
		if (host->irq)
			free_irq(host->irq, host);
		if (host->irq2)
			free_irq(host->irq2, host);
		if (host->base)
			iounmap(host->base);
		if (host->a2m_base)
			iounmap(host->a2m_base);
	}
	if (r)
		release_resource(r);
	if (mmc)
		mmc_free_host(mmc);
	return ret;
}

static int __exit pvcsd_remove(struct platform_device *pdev)
{
	struct mmc_host *mmc = platform_get_drvdata(pdev);

	if (mmc) {
		struct pvcsd_host *host = mmc_priv(mmc);
		mmc_remove_host(mmc);
		free_irq(host->irq, host);
		free_irq(host->irq2, host);
		del_timer_sync(&host->timer);
		pvcsd_power_down(host);
		iounmap(host->base);
		iounmap(host->a2m_base);
		release_resource(host->res);
		mmc_free_host(mmc);
	}
	platform_set_drvdata(pdev, NULL);
	return 0;
}

#ifdef CONFIG_PM
static int pvcsd_suspend(struct platform_device *dev, pm_message_t state)
{
	/* suspend/resume not supported */
	return -ENOSYS;
}

static int pvcsd_resume(struct platform_device *dev)
{
	/* suspend/resume not supported */
	return -ENOSYS;
}
#else
#define pvcsd_suspend	NULL
#define pvcsd_resume	NULL
#endif

#ifdef CONFIG_OF
static const struct of_device_id pvcsd_dt_ids[] = {
	{ .compatible = "panasonic,sdhci" },
	{ /* sentinel */ }
};

MODULE_DEVICE_TABLE(of, pvcsd_dt_ids);

static struct platform_driver pvcsd_driver = {
	.probe		= pvcsd_probe,
	.remove		= __exit_p(pvcsd_remove),
	.suspend	= pvcsd_suspend,
	.resume		= pvcsd_resume,
	.driver		= {
		.name	= DRIVER_NAME,
		.of_match_table = of_match_ptr(pvcsd_dt_ids),
	},
};
#else	/* CONFIG_OF */
static struct platform_driver pvcsd_driver = {
	.probe		= pvcsd_probe,
	.remove		= __exit_p(pvcsd_remove),
	.suspend	= pvcsd_suspend,
	.resume		= pvcsd_resume,
	.driver		= {
		.name	= DRIVER_NAME,
	},
};
#endif	/* CONFIG_OF */

static int __init pvcsd_init(void)
{
#ifdef PVCSD_ENABLE_CH
	if (ch < 0 || PVCSD_CH_MAX < ch) {
		ch = PVCSD_CH_MAX;
	}
#endif
	return platform_driver_probe(&pvcsd_driver, pvcsd_probe);
}

static void __exit pvcsd_exit(void)
{
	platform_driver_unregister(&pvcsd_driver);
}

module_init(pvcsd_init);
module_exit(pvcsd_exit);

module_param(maxfreq, int, 0);

/* force PIO transfers all the time */
module_param(nodma, int, 0);

#ifdef PVCSD_ENABLE_CH
module_param(ch, int, S_IRUGO);
#endif

MODULE_AUTHOR("");
MODULE_DESCRIPTION("Panasonic AVC MMC,SD,SDIO Host Controller driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:Panasonic AVC");

