/*
 * Suspend-to-RAM support code
 *
 * Copyright (C) 2018-2020 Panasonic Corporation
 *
 * This file is licensed under the terms of the GNU General Public
 * License version 2.  This program is licensed "as is" without any
 * warranty of any kind, whether express or implied.
 */

#include <linux/pm.h>
#include <linux/suspend.h>
#include <linux/module.h>
#include <linux/err.h>
#include <linux/cpu.h>

#include <asm/io.h>
#include <asm/system_misc.h>
#include <asm/suspend.h>

unsigned long* pvc_suspend_flag_area;

#define PVC_SUSPEND_RESUME_START_ADDR	(pvc_suspend_flag_area + 1)
#define PVC_SUSPEND_RESUME_FLG_ADDR	(pvc_suspend_flag_area + 2)
#define PVC_SUSPEND_RESUME_FLG_STR	"resume ok"

extern void pvc_sleep_disable_cache(void);

/*
 * sleep for resuming
 */
static int pvc_suspend_finish(unsigned long val)
{
	/* write resume entry point */
	*(PVC_SUSPEND_RESUME_START_ADDR) = virt_to_phys(cpu_resume);
	
	/* write resume ready */
	memcpy(PVC_SUSPEND_RESUME_FLG_ADDR, PVC_SUSPEND_RESUME_FLG_STR,
	       sizeof(PVC_SUSPEND_RESUME_FLG_STR));

	/* sleep */
	for (;;) {
		__asm__ __volatile__(
			"dsb\n\t"
			"wfi\n\t"
			: : : "memory"
		);
	}

	return 0;
}

/*
 * enter sleep state (suspend) and return on resuming
 */
static int pvc_suspend_enter(suspend_state_t suspend_state)
{
	/* enter suspend */
	cpu_suspend(0, pvc_suspend_finish);

	return 0;
}


static int pvc_suspend_begin(suspend_state_t state)
{
	cpu_idle_poll_ctrl(true);
	return 0;
}

static void pvc_suspend_end(void)
{
	cpu_idle_poll_ctrl(false);
	return;
}

/*
 * suspend callbacks
 */
struct platform_suspend_ops pvc_suspend_ops = {
	.begin		= pvc_suspend_begin,
	.end		= pvc_suspend_end,
	.enter		= pvc_suspend_enter,
	.valid		= suspend_valid_only_mem,
};

/*
 * initialize suspend
 */
int __init pvc_suspend_init(void)
{
	/* register callbacks */
	suspend_set_ops(&pvc_suspend_ops);

	/* map flag area */
	pvc_suspend_flag_area = ioremap_nocache(CONFIG_PVC_BOOT_FLG_AREA, SZ_4K);

	return 0;
}

__initcall(pvc_suspend_init);
