/*
 * Copyright (C) 2015-2020 Panasonic Corporation.
 *
 * This file is derived from arch/arm/plat-versatile/platsmp.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/io.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/jiffies.h>
#include <linux/smp.h>
#include <asm/cacheflush.h>
#include <asm/smp_plat.h>
#include <mach/board.h>

/*
 * Write pen_release in a way that is guaranteed to be visible to all
 * observers, irrespective of whether they're taking part in coherency
 * or not.  This is necessary for the hotplug code to work reliably.
 */
static void write_pen_release(int val)
{
	pen_release = val;
	smp_wmb();
	sync_cache_w(&pen_release);
}

static DEFINE_SPINLOCK(boot_lock);
extern void pvc_smp_secondary_startup(void);

static void __init pvc_smp_prepare_cpus(unsigned int max_cpus)
{
	/*
	 * Write the address of secondary startup into
	 * the synchronous flag. The boot monitor waits
	 * until it receives a soft interrupt, and then the
	 * secondary CPU branches to this address.
	 */
	writel(virt_to_phys(pvc_smp_secondary_startup), (void __iomem *)PVC_SMP_SYNC_VADDR);
}

static void pvc_smp_secondary_init(unsigned int cpu)
{
	/*
	 * if any interrupts are already enabled for the primary
	 * core (e.g. timer irq), then they will not have been enabled
	 * for us: do so
	 */
	printk("CPU%d: pvc_secondary_init\n", cpu);
	/*
	 * let the primary processor know we're out of the
	 * pen, then head off into the C entry point
	 */
	write_pen_release(-1);

	/*
	 * Synchronise with the boot thread.
	 */
	spin_lock(&boot_lock);
	spin_unlock(&boot_lock);

}

static int pvc_smp_boot_secondary(unsigned int cpu,
				    struct task_struct *idle)
{
	unsigned long timeout;

	/*
	 * Set synchronisation state between this boot processor
	 * and the secondary one
	 */
	spin_lock(&boot_lock);

	printk("CPU%d: boot_secondary...\n", cpu);

	write_pen_release(cpu_logical_map(cpu));

       /*
	 * Send the secondary CPU a soft interrupt, thereby causing
	 * the boot monitor to read the system wide flags register,
	 * and branch to the address found there.
	 */
	arch_send_wakeup_ipi_mask(cpumask_of(cpu));

	timeout = jiffies + (1 * HZ);
	while (time_before(jiffies, timeout)) {
		smp_rmb();
		if (pen_release == -1)
			break;

		udelay(10);
	}

	/*
	 * now the secondary core is starting up let it run its
	 * calibrations, then wait for it to finish
	 */
	spin_unlock(&boot_lock);

	return pen_release != -1 ? -ENOSYS : 0;
}

struct smp_operations pvc_smp_ops __initdata = {
	.smp_prepare_cpus	= pvc_smp_prepare_cpus,
	.smp_secondary_init	= pvc_smp_secondary_init,
	.smp_boot_secondary	= pvc_smp_boot_secondary,
};
CPU_METHOD_OF_DECLARE(pvc_smp, "panasonic,pvc", &pvc_smp_ops);
