/*
 *  linux/arch/arm/common/timer-sp.c
 *
 *  Copyright (C) 1999 - 2003 ARM Limited
 *  Copyright (C) 2000 Deep Blue Solutions Ltd
 *  Copyright (C) 2017-2020 Panasonic Corporation.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/clk.h>
#include <linux/clocksource.h>
#include <linux/clockchips.h>
#include <linux/err.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/sched_clock.h>

#ifdef CONFIG_ARCH_PVC
#include <linux/smp.h>
#include <linux/cpu.h>
#endif

#include <asm/hardware/arm_timer.h>
#include <asm/hardware/timer-sp.h>

static long __init sp804_get_clock_rate(struct clk *clk)
{
	long rate;
	int err;

	err = clk_prepare(clk);
	if (err) {
		pr_err("sp804: clock failed to prepare: %d\n", err);
		clk_put(clk);
		return err;
	}

	err = clk_enable(clk);
	if (err) {
		pr_err("sp804: clock failed to enable: %d\n", err);
		clk_unprepare(clk);
		clk_put(clk);
		return err;
	}

	rate = clk_get_rate(clk);
	if (rate < 0) {
		pr_err("sp804: clock failed to get rate: %ld\n", rate);
		clk_disable(clk);
		clk_unprepare(clk);
		clk_put(clk);
	}

	return rate;
}

static void __iomem *sched_clock_base;

static u64 notrace sp804_read(void)
{
	return ~readl_relaxed(sched_clock_base + TIMER_VALUE);
}

void __init __sp804_clocksource_and_sched_clock_init(void __iomem *base,
						     const char *name,
						     struct clk *clk,
						     int use_sched_clock)
{
	long rate;

	if (!clk) {
		clk = clk_get_sys("sp804", name);
		if (IS_ERR(clk)) {
			pr_err("sp804: clock not found: %d\n",
			       (int)PTR_ERR(clk));
			return;
		}
	}

	rate = sp804_get_clock_rate(clk);

	if (rate < 0)
		return;

	/* setup timer 0 as free-running clocksource */
	writel(0, base + TIMER_CTRL);
	writel(0xffffffff, base + TIMER_LOAD);
	writel(0xffffffff, base + TIMER_VALUE);
	writel(TIMER_CTRL_32BIT | TIMER_CTRL_ENABLE | TIMER_CTRL_PERIODIC,
		base + TIMER_CTRL);

	clocksource_mmio_init(base + TIMER_VALUE, name,
		rate, 200, 32, clocksource_mmio_readl_down);

	if (use_sched_clock) {
		sched_clock_base = base;
		sched_clock_register(sp804_read, 32, rate);
	}
}

#ifdef CONFIG_ARCH_PVC
struct sp804_timer_device {
	void __iomem *base;
	unsigned long reload;
	long rate;
	struct clock_event_device evt;
	struct irqaction irqaction;
};

static struct sp804_timer_device __percpu *local_dev;
#else
static void __iomem *clkevt_base;
static unsigned long clkevt_reload;
#endif

/*
 * IRQ handler for the timer
 */
static irqreturn_t sp804_timer_interrupt(int irq, void *dev_id)
{
	struct clock_event_device *evt = dev_id;
#ifdef CONFIG_ARCH_PVC
	struct sp804_timer_device *device = container_of(evt, struct sp804_timer_device, evt);
	void __iomem *clkevt_base = device->base;
#endif

	/* clear the interrupt */
	writel(1, clkevt_base + TIMER_INTCLR);

	evt->event_handler(evt);

	return IRQ_HANDLED;
}

static void sp804_set_mode(enum clock_event_mode mode,
	struct clock_event_device *evt)
{
#ifdef CONFIG_ARCH_PVC
	struct sp804_timer_device *device = container_of(evt, struct sp804_timer_device, evt);
	void __iomem *clkevt_base = device->base;
	unsigned long clkevt_reload = device->reload;
#endif
	unsigned long ctrl = TIMER_CTRL_32BIT | TIMER_CTRL_IE;

	writel(ctrl, clkevt_base + TIMER_CTRL);

	switch (mode) {
	case CLOCK_EVT_MODE_PERIODIC:
		writel(clkevt_reload, clkevt_base + TIMER_LOAD);
		ctrl |= TIMER_CTRL_PERIODIC | TIMER_CTRL_ENABLE;
		break;

	case CLOCK_EVT_MODE_ONESHOT:
		/* period set, and timer enabled in 'next_event' hook */
		ctrl |= TIMER_CTRL_ONESHOT;
		break;

	case CLOCK_EVT_MODE_UNUSED:
	case CLOCK_EVT_MODE_SHUTDOWN:
	default:
		break;
	}

	writel(ctrl, clkevt_base + TIMER_CTRL);
}

static int sp804_set_next_event(unsigned long next,
	struct clock_event_device *evt)
{
#ifdef CONFIG_ARCH_PVC
	struct sp804_timer_device *device = container_of(evt, struct sp804_timer_device, evt);
	void __iomem *clkevt_base = device->base;
#endif
	unsigned long ctrl = readl(clkevt_base + TIMER_CTRL);

	writel(next, clkevt_base + TIMER_LOAD);
	writel(ctrl | TIMER_CTRL_ENABLE, clkevt_base + TIMER_CTRL);

	return 0;
}

#ifdef CONFIG_ARCH_PVC
#ifdef CONFIG_SUSPEND
static void sp804_timer_suspend(struct clock_event_device *evt)
{
	/* stop timer */
	writel(0, sched_clock_base + TIMER_CTRL);
}

static void sp804_timer_resume(struct clock_event_device *evt)
{
	/* setup timer */
	writel(0, sched_clock_base + TIMER_CTRL);
	writel(0xffffffff, sched_clock_base + TIMER_LOAD);
	writel(0xffffffff, sched_clock_base + TIMER_VALUE);
	writel(TIMER_CTRL_32BIT | TIMER_CTRL_ENABLE | TIMER_CTRL_PERIODIC,
	       sched_clock_base + TIMER_CTRL);
}
#endif

static struct sp804_timer_device default_dev = {
	.evt = {
		.features       = CLOCK_EVT_FEAT_PERIODIC | CLOCK_EVT_FEAT_ONESHOT |
		CLOCK_EVT_FEAT_DYNIRQ,
		.set_mode	= sp804_set_mode,
		.set_next_event	= sp804_set_next_event,
		.rating		= 300,
#ifdef CONFIG_SUSPEND
		.suspend	= sp804_timer_suspend,
		.resume		= sp804_timer_resume,
#endif
	},
	.irqaction = {
		.name		= "timer",
		.flags		= IRQF_TIMER | IRQF_IRQPOLL,
		.handler	= sp804_timer_interrupt,
		.dev_id		= &default_dev.evt,
	},
};
#else
static struct clock_event_device sp804_clockevent = {
	.features       = CLOCK_EVT_FEAT_PERIODIC | CLOCK_EVT_FEAT_ONESHOT |
		CLOCK_EVT_FEAT_DYNIRQ,
	.set_mode	= sp804_set_mode,
	.set_next_event	= sp804_set_next_event,
	.rating		= 300,
};

static struct irqaction sp804_timer_irq = {
	.name		= "timer",
	.flags		= IRQF_TIMER | IRQF_IRQPOLL,
	.handler	= sp804_timer_interrupt,
	.dev_id		= &sp804_clockevent,
};
#endif /* !CONFIG_ARCH_PVC */

void __init __sp804_clockevents_init(void __iomem *base, unsigned int irq, struct clk *clk, const char *name)
{
#ifdef CONFIG_ARCH_PVC
	struct clock_event_device *evt = &default_dev.evt;
#else
	struct clock_event_device *evt = &sp804_clockevent;
#endif
	long rate;

	if (!clk)
		clk = clk_get_sys("sp804", name);
	if (IS_ERR(clk)) {
		pr_err("sp804: %s clock not found: %d\n", name,
			(int)PTR_ERR(clk));
		return;
	}

	rate = sp804_get_clock_rate(clk);
	if (rate < 0)
		return;

#ifdef CONFIG_ARCH_PVC
	default_dev.base = base;
	default_dev.reload = DIV_ROUND_CLOSEST(rate, HZ);
#else
	clkevt_base = base;
	clkevt_reload = DIV_ROUND_CLOSEST(rate, HZ);
#endif
	evt->name = name;
	evt->irq = irq;
	evt->cpumask = cpu_possible_mask;

	writel(0, base + TIMER_CTRL);

#ifdef CONFIG_ARCH_PVC
	setup_irq(irq, &default_dev.irqaction);
#else
	setup_irq(irq, &sp804_timer_irq);
#endif
	clockevents_config_and_register(evt, rate, 0xf, 0xffffffff);
}

#ifdef CONFIG_ARCH_PVC
/*
 * Setup the local clock events for a CPU.
 */
static void sp804_local_timer_setup(void)
{
	struct sp804_timer_device *device = raw_cpu_ptr(local_dev);
	int cpu = smp_processor_id();

	/*
	 * The following is done once per CPU the first time .setup() is
	 * called.
	 */
	device->evt.name = "local_timer";
	device->evt.features = CLOCK_EVT_FEAT_PERIODIC | CLOCK_EVT_FEAT_ONESHOT |
		CLOCK_EVT_FEAT_C3STOP;
	device->evt.rating = 300;
	device->evt.set_mode = sp804_set_mode;
	device->evt.set_next_event = sp804_set_next_event;
	device->evt.cpumask = cpumask_of(cpu);

	clockevents_config_and_register(&device->evt, device->rate,
					0xf, 0xffffffff);

	device->irqaction.name = "local_timer";
	device->irqaction.flags = IRQF_TIMER | IRQF_IRQPOLL;
	device->irqaction.handler = sp804_timer_interrupt;
	device->irqaction.dev_id = &device->evt;

	setup_irq(device->evt.irq, &device->irqaction);
	irq_force_affinity(device->evt.irq, cpumask_of(cpu));
}

static void sp804_local_timer_stop(void)
{
	struct sp804_timer_device *timer= raw_cpu_ptr(local_dev);

	writel(0, timer->base + TIMER_CTRL);
	disable_irq(timer->evt.irq);
}

static int sp804_local_timer_cpu_notify(struct notifier_block *self,
				unsigned long action, void *hcpu)
{
	switch (action & ~CPU_TASKS_FROZEN) {
	case CPU_STARTING:
		sp804_local_timer_setup();
		break;
	case CPU_DYING:
		sp804_local_timer_stop();
		break;
	}

	return NOTIFY_OK;
}

static struct notifier_block sp804_local_timer_cpu_nb = {
	.notifier_call = sp804_local_timer_cpu_notify,
};

static void __init __sp804_local_timer_init(struct sp804_timer_device *device, void *base, int irq, long rate)
{
	device->base = base;
	device->evt.irq = irq;
	device->rate = rate;
	device->reload = DIV_ROUND_CLOSEST(rate, HZ);
}

static int __init sp804_local_timer_of_init(struct device_node *np, void *base)
{
	static int init_cpu = 0;
	struct clk *clk1, *clk2;
	int irq1, irq2, err;
	long rate;

	clk1 = of_clk_get(np, 0);
	if (IS_ERR(clk1))
		clk1 = NULL;

	/* Get the 2nd clock if the timer has 3 timer clocks */
	if (of_count_phandle_with_args(np, "clocks", "#clock-cells") == 3) {
		clk2 = of_clk_get(np, 1);
		if (IS_ERR(clk2)) {
			pr_err("sp804: %s clock not found: %d\n", np->name,
				(int)PTR_ERR(clk2));
			clk2 = NULL;
		}
	} else
		clk2 = clk1;

	if (init_cpu == 0) {
		/* If this is the first time, alocate structure for local timer */
		local_dev = alloc_percpu(struct sp804_timer_device);
		if (!local_dev) {
			err = -ENOMEM;
			goto out;
		}

		err = register_cpu_notifier(&sp804_local_timer_cpu_nb);
		if (err) {
			goto out;
		}
	}

	if (init_cpu < NR_CPUS) {
		irq1 = irq_of_parse_and_map(np, 0);
		if (irq1 <= 0) {
			err = -EINVAL;
			goto out;
		}

		rate = sp804_get_clock_rate(clk1);
		if (rate < 0) {
			err = -EINVAL;
			goto out;
		}

		__sp804_local_timer_init(per_cpu_ptr(local_dev, init_cpu), base, irq1, rate);
		if (init_cpu == 0) {
			/* Setup local timer for CPU0 */
			sp804_local_timer_setup();
		}
		init_cpu++;
	}

	if (init_cpu < NR_CPUS) {
		irq2 = irq_of_parse_and_map(np, 1);
		if (irq2 <= 0) {
			err = -EINVAL;
			goto out;
		}

		rate = sp804_get_clock_rate(clk2);
		if (rate < 0) {
			err = -EINVAL;
			goto out;
		}

		__sp804_local_timer_init(per_cpu_ptr(local_dev, init_cpu), base + TIMER_2_BASE, irq2, rate);
		init_cpu++;
	}
	return 0;
out:
	if (init_cpu == 0 && local_dev) {
		free_percpu(local_dev);
		local_dev = NULL;
	}

	return err;
}
#endif /* CONFIG_ARCH_PVC */

static void __init sp804_of_init(struct device_node *np)
{
	static bool initialized = false;
	void __iomem *base;
	int irq;
	u32 irq_num = 0;
	struct clk *clk1, *clk2;
	const char *name = of_get_property(np, "compatible", NULL);

	base = of_iomap(np, 0);
	if (WARN_ON(!base))
		return;

	/* Ensure timers are disabled */
	writel(0, base + TIMER_CTRL);
	writel(0, base + TIMER_2_BASE + TIMER_CTRL);

#ifdef CONFIG_ARCH_PVC
	if (of_get_property(np, "arm,sp804-local", NULL) && of_device_is_available(np)) {
		if (sp804_local_timer_of_init(np, base) < 0) {
			pr_err("sp804: can't register local timer\n");
			goto err;
		}
		return;
	}
#endif /* CONFIG_ARCH_PVC */
	if (initialized || !of_device_is_available(np))
		goto err;

	clk1 = of_clk_get(np, 0);
	if (IS_ERR(clk1))
		clk1 = NULL;

	/* Get the 2nd clock if the timer has 3 timer clocks */
	if (of_count_phandle_with_args(np, "clocks", "#clock-cells") == 3) {
		clk2 = of_clk_get(np, 1);
		if (IS_ERR(clk2)) {
			pr_err("sp804: %s clock not found: %d\n", np->name,
				(int)PTR_ERR(clk2));
			clk2 = NULL;
		}
	} else
		clk2 = clk1;

	irq = irq_of_parse_and_map(np, 0);
	if (irq <= 0)
		goto err;

	of_property_read_u32(np, "arm,sp804-has-irq", &irq_num);
	if (irq_num == 2) {
		__sp804_clockevents_init(base + TIMER_2_BASE, irq, clk2, name);
		__sp804_clocksource_and_sched_clock_init(base, name, clk1, 1);
	} else {
		__sp804_clockevents_init(base, irq, clk1 , name);
		__sp804_clocksource_and_sched_clock_init(base + TIMER_2_BASE,
							 name, clk2, 1);
	}
	initialized = true;

	return;
err:
	iounmap(base);
}
CLOCKSOURCE_OF_DECLARE(sp804, "arm,sp804", sp804_of_init);

static void __init integrator_cp_of_init(struct device_node *np)
{
	static int init_count = 0;
	void __iomem *base;
	int irq;
	const char *name = of_get_property(np, "compatible", NULL);
	struct clk *clk;

	base = of_iomap(np, 0);
	if (WARN_ON(!base))
		return;
	clk = of_clk_get(np, 0);
	if (WARN_ON(IS_ERR(clk)))
		return;

	/* Ensure timer is disabled */
	writel(0, base + TIMER_CTRL);

	if (init_count == 2 || !of_device_is_available(np))
		goto err;

	if (!init_count)
		__sp804_clocksource_and_sched_clock_init(base, name, clk, 0);
	else {
		irq = irq_of_parse_and_map(np, 0);
		if (irq <= 0)
			goto err;

		__sp804_clockevents_init(base, irq, clk, name);
	}

	init_count++;
	return;
err:
	iounmap(base);
}
CLOCKSOURCE_OF_DECLARE(intcp, "arm,integrator-cp-timer", integrator_cp_of_init);
