/*
 *  Copyright Altera Corporation (C) 2012-2013. All rights reserved
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms and conditions of the GNU General Public License,
 *  version 2, as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 *  more details.
 *
 *  You should have received a copy of the GNU General Public License along with
 *  this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <asm/io.h>
#include <nand.h>
#include <watchdog.h>

DECLARE_GLOBAL_DATA_PTR;

/*
 * Print Board information
 */
int checkboard(void)
{
#ifdef CONFIG_SOCFPGA_VIRTUAL_TARGET
	puts("BOARD : Altera VTDEV5XS1 Virtual Board\n");
#else
	puts("BOARD : Altera SOCFPGA Cyclone V Board\n");
#endif
	return 0;
}

/* ///////////////////////////////////////////////////////////////// *
 * Diag Func
 * ///////////////////////////////////////////////////////////////// */
struct ddr_result{
	unsigned char result;			/* OK/NG */
	unsigned int address;			/* target address */
	unsigned int expect_value;
	unsigned int actual_value;
};
static unsigned int rtc_time = 0;

#define TC_OSC0                         ((unsigned long)0xFFD00000)
#define TC_OSC1                         ((unsigned long)0xFFD01000)
#define TC_SP0                          ((unsigned long)0xFFC08000)
#define TC_SP1                          ((unsigned long)0xFFC09000)

#define RTC                             TC_OSC1
#define TC0		                TC_SP0  /* not used */
#define TC1		                TC_SP1

#define UART0                           ((unsigned long)0xFFC02000)
#define UART1                           ((unsigned long)0xFFC03000)

#ifndef CONFIG_DIAG_DEBUG
#define USARTn                          UART1
#define USART_FOR_LOG                   "eserial0"
#define USART_FOR_DIAG                  "eserial1"
#else  
#define USARTn                          UART0
#define USART_FOR_LOG                   "eserial0" /* same for debug */
#define USART_FOR_DIAG                  "eserial0"
#endif

/* for TC */
#define TIME_10ms	                (1000000)  /* (1 / cm_get_l4_sp_clk_hz()) */  /*0x00002849  */  /* 10ms * (132MHz / 128) */
#define TIME_20ms	                (2000000)  /* (1 / cm_get_l4_sp_clk_hz()) */  /*0x00005091  */  /* 20ms * (132MHz / 128) */

#define RTC_30sec                       (0x2CB4177F)
/* for RTC */
#define MODE1_LIMIT_TIME	        30	        /* 30 Minute */

/* for protocol */
#define MSG_STX				0x02	        /* STX value */
#define MSG_BOAD_TYPE_NW		0x06	        /* Boad Type  : NW-Board */
#define MSG_CHECK_TYPE_INPUT	        0x01 	        /* Check Type: Input test */
#define MSG_CHECK_TYPE_OUTPUT	        0x02 	        /* Check Type: Output test */
#define MSG_CHECK_TYPE_OTHER	        0x04 	        /* Check Type: Other test */

#define DATA_LENGTH_MAX	                3	        /* Non OS Self Check Mode: Maximum data field length */

#define PROC_NO_MIN		        3	        /* Non OS Self Check Mode: Minimum process number */
#define PROC_NO_MAX		        35	        /* Non OS Self Check Mode: Maximum process number */

#define RXRDY_DETECT	                0x1
#define RXRDY_NON		        0x0

#define FC_NW_SELF_CHECK			        /* Self Check mode */

/*
 * checking to DDR data bus & address bus
 */

#define RET_OK				0		/* Result PASS */
#define RET_NG				1		/* Result FAIL */
#define RETURN_VALUE_NONE	        2		/* Result value none */
#define RET_NG_NOT_FOUND	        3		/* PHY not found */
#define RET_NG_RX			4		/* Abort (RXRDY Detect) */

#define AT91C_BASE_DDRCS		0x00000000
#define DDR_SIZE			0x07E00000
#define DDR_VERIFY_PATTERN1		0x55555555
#define DDR_VERIFY_PATTERN2		0xaaaaaaaa

#define RET_ABORT			RET_NG_RX


#define TC_CTRL                         0x08UL
#define TC_EOI                          0x0CUL
#define TC_LDCNT                        0x00UL
#define TC_CURCNT                       0x04UL
#define TC_RAWINT                       0xA8UL

#define USARTn_DLL                      0x00UL
#define USARTn_DLH                      0x04UL
#define USARTn_FCR                      0x08UL
#define USARTn_FMT                      0x0CUL
#define USARTn_MCR                      0x10UL
#define USARTn_LSR                      0x14UL
#define USARTn_FIFO                     0x88UL
#define USARTn_SFE                      0x98UL

#define AT91C_USART_PARE                0x00000004UL
#define AT91C_USART_FRAME               0x00000008UL
#define AT91C_USART_OVRE                0x00000002UL


static void diag_main(void);
static unsigned int uiSelfCheck_limit_set(unsigned int limit_min);
static unsigned int uiRTC_GetTimeSec(void);
static unsigned char ucUSART3_get_char    (unsigned char *data);
static void vTC1_init(void);
static void vTC_start (unsigned int channel);
static void vTC_stop  (unsigned int channel);
static void vTC_reset (unsigned int channel);
static unsigned int  uiTC_get_value(unsigned int channel);

static void vSelfCheck_Mode_time(unsigned int limit);
static unsigned char ucUSART3_RXRDY_detect(void);

static unsigned char ucDIAG_telegram_check(unsigned char *buf, unsigned char *test_type, unsigned char *proc_number);
static void output_hex8(unsigned char byte);
static void output_hex_8bit(unsigned char byte);
static void output_hex(unsigned int num);
static unsigned char ucDIAG_calculateBCC  (unsigned char *frame, unsigned char data_length);
static void vUSART3_hw_init      (void);
static void vUSART3_send_telegram(unsigned char *buf);
static void usart_puts(const char *ptr);
static char usart_getc(void);
static void output_hex8(unsigned char byte);
static void output_hex_8bit(unsigned char byte);
static void output_hex(unsigned int num);
static void vBoardReset(void);

static inline unsigned int read_tc(unsigned int offset)
{
        return readl(offset);
}


static inline void write_tc(unsigned int offset, const unsigned int value)
{
	writel(value, offset);
}


static inline void write_usart3(unsigned int offset, const unsigned int value)
{
	writel(value, offset);
}


static inline unsigned int read_usart3(unsigned int offset)
{
	return readl(offset);
}

static inline unsigned int read_ddr(unsigned int offset)
{
	return readl(offset + AT91C_BASE_DDRCS);
}

static inline void write_ddr(unsigned int offset, const unsigned int value)
{
	writel(value, offset + AT91C_BASE_DDRCS);
}

/* ////////////////////////////////////////////////////////////////////////////////////////////// */
/*
 * Init GPIO
 */
int gpio_init(void)
{
  int ret=0;

  /* printf("GPIO0 DR  = 0x%08x\n", SOCFPGA_GPIO0_DR); */
  writel(SOCFPGA_GPIO0_DR, SOCFPGA_GPIO0_ADDR + SOCFPGA_GPIO_DR);

  /* printf("GPIO0 DDR = 0x%08x\n", SOCFPGA_GPIO0_DDR); */
  writel(SOCFPGA_GPIO0_DDR, SOCFPGA_GPIO0_ADDR + SOCFPGA_GPIO_DDR);
  
  /* printf("GPIO1 DR  = 0x%08x\n", SOCFPGA_GPIO1_DR); */
  writel(SOCFPGA_GPIO1_DR, SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);

  /* printf("GPIO1 DDR = 0x%08x\n", SOCFPGA_GPIO1_DDR); */
  writel(SOCFPGA_GPIO1_DDR, SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DDR);
  
  /* printf("GPIO2 DR  = 0x%08x\n", SOCFPGA_GPIO2_DR); */
  writel(SOCFPGA_GPIO2_DR, SOCFPGA_GPIO2_ADDR + SOCFPGA_GPIO_DR);

  /* printf("GPIO2 DDR = 0x%08x\n", SOCFPGA_GPIO2_DDR); */
  writel(SOCFPGA_GPIO2_DDR, SOCFPGA_GPIO2_ADDR + SOCFPGA_GPIO_DDR);

  return ret;
}

#define BOOT_SWITCH_ADDR	(0x00360000)
#define BOOT_SWITCH_BLOCK_NUM	(3)

#define BLOCK_SIZE		CONFIG_SYS_NAND_BLOCK_SIZE
#define BOOT_SWITCH_LEN		(1)

#define BOOT_MAGICNUM_ADDR	(0x003C0000)
#define BOOT_MAGICNUM_BLOCK_NUM	(4)
#define BOOT_MAGICNUM_LEN	(1)

/*
 * Swich Boot Bank
 */
static int nand_read_flag(nand_info_t *nand, loff_t off, size_t *len, int num, uint8_t *buffer)
{
  int ret, rd_block = 0;


  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  while (rd_block < num){
    
    loff_t rd_off = off + (rd_block * BLOCK_SIZE);

    ret = nand_block_isbad(nand, rd_off);
    if (ret < 0){
      printf("NAND ERROR: Discovered badblock!:%d\n", ret);
      rd_block++;
      continue;
    }

    ret = nand_read(nand, rd_off, len,  buffer);
    if(ret < 0){
      printf("NAND ERROR: Read fail!:%d\n", ret);
      rd_block++;
      continue;
    }

    return 0;
  }
  
  memset(buffer, 0xff, len);
  return -1;
}

static int nand_write_flag(nand_info_t *nand, loff_t off, size_t *len, int num, uint8_t *buffer)
{
  int ret, rd_block = 0;

  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  while (rd_block < num){
    
    loff_t rd_off = off + (rd_block * BLOCK_SIZE);

    ret = nand_block_isbad(nand, rd_off);
    if (ret < 0){
      printf("NAND ERROR: Discovered badblock!:%d\n", ret);
      rd_block++;
      continue;
    }

    ret = nand_erase(nand, rd_off, BLOCK_SIZE);
    if(ret < 0){
      printf("NAND ERROR: Erase fail!:%d\n", ret);
      rd_block++;
      continue;
    }

#if 0 /* for debug --->*/
    for (ret = 0; ret < *len; ret++){
      printf("***** buffer[%d] = %02x\n", ret, buffer[ret]);
    }
#endif /* <--- for debug */
    
    ret = nand_write(nand, rd_off, len,  buffer);
    if(ret < 0){
      printf("NAND ERROR: Write fail!:%d\n", ret);
    }
    /* printf("***** rd_block = %d\n", rd_block); */

    rd_block++;
  }
  
  return 0;
}

static int switch_bank(int bank)
{

  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  switch(bank){
  case 1:
  default:
    setenv("bootcmd", "run boot_swa; run nandload; run nandboot_ef");
    printf("***** bootcmd = run boot_swa; run nandload; run nandboot_ef\n");
    break;
  case 2:
    setenv("bootcmd", "run boot_swb; run nandload; run nandboot_ef");
    printf("***** bootcmd = run boot_swb; run nandload; run nandboot_ef\n");
    break;
  }
}

int switch_bootbank(void)
{
  int ret, btbank = 0;
  u_char bootsw, magic;
  nand_info_t *nand = &nand_info[0];
  size_t len;
  u_char buffer[2] = {0xff, 0xff};
  
  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  len = BOOT_SWITCH_LEN;
  ret = nand_read_flag(nand, BOOT_SWITCH_ADDR, &len, BOOT_SWITCH_BLOCK_NUM, buffer);
  bootsw = buffer[0];
  if(ret < 0){
    printf("ERROR: Was broken boot switch data!!\n");
  }
  printf("***** bootsw = %02x\n", bootsw);

  len =  BOOT_MAGICNUM_LEN;
  ret = nand_read_flag(nand, BOOT_MAGICNUM_ADDR, &len, BOOT_MAGICNUM_BLOCK_NUM, buffer);
  magic = buffer[0];
  if(ret < 0){
    printf("ERROR: Was broken boot magic number!!\n");
  }
  printf("***** magic = %02x\n", magic);

  switch (bootsw) {
  case 0x01:
  case 0x02:
    btbank = 1;
    break;
    
  case 0x03:
    if(magic == 0x01 | magic == 0x02){
      btbank = 1;
    }else if(magic == 0x00 | magic == 0xff){
      btbank = 2;
    }else {
      btbank = 2;
    }
    break;
    
  case 0x11:
  case 0x12:
    btbank = 2;
    break;
    
  case 0x13:
    if(magic == 0x01 | magic == 0x02){
      btbank = 2;
    }else if(magic == 0x00 | magic == 0xff){
      btbank = 1;
    }else {
      btbank = 1;
    }
    break;

  default:
    if(magic == 0x03){
      btbank = 2;
      magic = 0x04;
    }else {
      btbank = 1;
      magic = 0x03;
    }
    break;
  }

  switch_bank(btbank);

  if(magic == 0xff)
    magic = 0;

  if((magic < 0x02) || (magic == 0x03) || (magic == 0x04)) {
    buffer[0] = (magic < 0x02) ? (magic + 1) : magic;
    /* printf("***** magic = %02x, buffer[0] = %02x\n", magic, buffer[0]); */
    len =  BOOT_MAGICNUM_LEN;
    nand_write_flag(nand, BOOT_MAGICNUM_ADDR, &len, BOOT_MAGICNUM_BLOCK_NUM, buffer);
  }
  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  
  return 0;
}


/*
 * Ether Intrrupt Check
 */
unsigned char ucEth_Int_Check(void)
{
  unsigned int val;
  const char *devname = "mii0";

  /* Read MISR1  */
  miiphy_read(devname, CONFIG_EPHY0_PHY_ADDR, 0x12, &val);
  printf("***** MII MISR1 = 0x%04x\n", val & 0xffff);
  /* Read MISR2  */
  miiphy_read(devname, CONFIG_EPHY0_PHY_ADDR, 0x13, &val);
  printf("***** MII MISR2 = 0x%04x\n", val & 0xffff);

  udelay(1000);
  
  /* INPUT GPIO56(GPIO1[27]) */
  val =  readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_EXT_PORTA);
  if ( !(val & BIT(27)) ){
    printf("***** Pre check ETH Interrupt Level: ERROR!!(INT Level is Low.)\n");
    return RET_NG;
  }

  /* Setting TI DP83822 0x0012: RMII Interrupt Status Register #1(MIISR1) */
  /* Register 0x12.5: Link Status Changed Enable */
  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  miiphy_read(devname, CONFIG_EPHY0_PHY_ADDR, 0x12, &val);
  miiphy_write(devname, 1, 0x12, val | BIT(5));

  /* Setting TI DP83822 0x0011: PHY Specific Control Register (PHYSCR) */
  /* Register 0x11.: Link Status Changed Enable */
  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  miiphy_read(devname, CONFIG_EPHY0_PHY_ADDR, 0x11, &val);
  miiphy_write(devname, 1, 0x11, (val & 0xff00) | BIT(8) | BIT(3) | BIT(2) | BIT(1) | BIT(0));
  
  mdelay(500);

  /* INPUT GPIO56(GPIO1[27]) */
  val =  readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_EXT_PORTA);
  if ( val & BIT(27) ){
    printf("***** Post check ETH Interrupt Level: ERROR!!(INT Level is High.)\n");
    return RET_NG;
  }
  printf("***** Post check ETH Interrupt Level: OK(INT Level is Low.)\n");
  
  /* Setting TI DP83822 0x0011: PHY Specific Control Register (PHYSCR) */
  /* Register 0x11.: Link Status Changed Enable */
  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  miiphy_read(devname, CONFIG_EPHY0_PHY_ADDR, 0x11, &val);
  miiphy_write(devname, 1, 0x11, (val & 0xff00) | BIT(8) | BIT(3) | BIT(1) | BIT(0));
  
  return RET_OK;
}


/*
 * LPWA Output Check
 */
unsigned char ucLPWA_Output(unsigned char proc_number)
{
  unsigned int val;
  unsigned char ret = 0;

  switch (proc_number) {
  case 0x1E:
    /* RESET_N ON */
    val = readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    writel(val | BIT(15), SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    /* WAKE_UP ON */
    val = readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    writel(val | BIT(17), SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    /* UART_RTS ON */
    val = readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    writel(val | BIT(0), SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    ret = 1;
    break;
  case 0x1F:
    /* RESET_N OFF */
    val = readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    writel(val & ~BIT(15), SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    /* WAKE_UP OFF */
    val = readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    writel(val & ~BIT(17), SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    /* UART_RTS OFF */
    val = readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    writel(val & ~BIT(0), SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_DR);
    ret = 0;
    break;
  default:
    return RET_NG;
  }

  return RET_OK;
}


/*
 * LPWA Input Check
 */
unsigned char ucLPWA_Input(unsigned char proc_number)
{
  unsigned int val;
  unsigned char ret = 0;

  switch (proc_number) {
  case 0x20:	/* HPD ON */
  case 0x22:	/* HPD OFF */
    val = readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_EXT_PORTA);
    if ( val & BIT(18) ) {
      ret = 1;
    } else {
      ret = 0;
    }
    break;
  case 0x21:	/* UART_CTS ON */
  case 0x23:	/* UART_CTS OFF */
    val = readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_EXT_PORTA);
    if ( val & BIT(16) ) {
      ret = 1;
    } else {
      ret = 0;
    }
    break;
  default:
    break;
  }

  return ret;
}


/*-----------------------------------------------------------------------------
 * DDR2 Check Verify Error Infomation output
 *    RETRUN value : none
 *    struct ddr_result *chk_result : check result info
 *---------------------------------------------------------------------------*/
static void output_error_info(struct ddr_result *chk_result)
{
#if 0 /* original code */
	usart_puts(" - Verify error. Address: ");
	output_hex(chk_result->address);
	usart_puts(", Data: ");
	output_hex(chk_result->actual_value);
	usart_puts("\n");
#else
	printf(" - Verify error. Address: %08x, Data: %08x\n", chk_result->address, chk_result->actual_value);
#endif
}

/*-----------------------------------------------------------------------------
 * DDR2 Fix Pattern Verify Check
 *    RETRUN value : (unsigned char) RET_OK, RET_NG, RET_NG_RX(RXRDY detect)
 *    struct ddr_result *chk_result : check result info
 *    unsigned int pattern          : check pattern(32bit)
 *---------------------------------------------------------------------------*/
static unsigned char ucDDR_verify(struct ddr_result *chk_result, unsigned int pattern)
{
	unsigned int addr = 0;
	unsigned int data = 0;
	
	chk_result->expect_value = pattern;
	
#if ENABLE_INTRRUPT_COMANND /* Interrupt command check */
	/* Check Receiver Ready */
	if (ucUSART3_RXRDY_detect() == RXRDY_DETECT)
	{
		chk_result->result = RET_NG_RX;
		return RET_NG;
	}
	usart_puts(" Test Pattern ( ");
	output_hex(pattern);
	usart_puts(" ) Write...");
#endif
	for (addr = 0; addr < DDR_SIZE; addr += 4)
	{
		write_ddr(addr, pattern);
	}
	uiRTC_GetTimeSec();
#ifdef CONFIG_HW_WATCHDOG
	WATCHDOG_RESET();
#endif

#if ENABLE_INTRRUPT_COMANND /* Interrupt command check */
	/* Check Receiver Ready */
	if (ucUSART3_RXRDY_detect() == RXRDY_DETECT)
	{
		chk_result->result = RET_NG_RX;
		return RET_NG;
	}
	usart_puts("\n Read...\n");
#endif
	for(addr = 0; addr < DDR_SIZE; addr += 4)
	{
		data = read_ddr(addr);
		if (data != pattern)
		{
			chk_result->result       = RET_NG;
			chk_result->address      = AT91C_BASE_DDRCS + addr;
			chk_result->actual_value = data;
			uiRTC_GetTimeSec();
#ifdef CONFIG_HW_WATCHDOG
			WATCHDOG_RESET();
#endif
			return RET_NG;
		}
	}
	uiRTC_GetTimeSec();
#ifdef CONFIG_HW_WATCHDOG
	WATCHDOG_RESET();
#endif
	return RET_OK;
}


/*-----------------------------------------------------------------------------
 * DDR2 Fix Pattern Write/Readd Check
 *    RETRUN value : (unsigned char) RET_OK, RET_NG, RET_NG_RX(RXRDY detect)
 *---------------------------------------------------------------------------*/
unsigned char ucDDR2_ReadWrite(void)
{
	unsigned int addr;
	struct ddr_result ret = { RET_OK, AT91C_BASE_DDRCS, 0x00000000, 0x00000000};

#if ENABLE_INTRRUPT_COMANND /* Interrupt command check */
	/* Check Receiver Ready */
	if (ucUSART3_RXRDY_detect() == RXRDY_DETECT)
	{
		return RET_NG_RX;
	}
	/* Fill 0xff */
	usart_puts("Fill 0xFFFFFFFF... ");
#else
	printf("Fill 0xffff_ffff ...");
#endif
	for (addr = 0; addr < DDR_SIZE; addr += 4)
	{
		write_ddr(addr, 0xFFFFFFFF);
	}
	uiRTC_GetTimeSec();
#ifdef CONFIG_HW_WATCHDOG
	WATCHDOG_RESET();
#endif
#if 0 /* original code */
	usart_puts(" Complete.\n");

	/* Verify Check 1 */
	usart_puts("Verify Check [TestPattern1] \n");
#else
	printf(" Complete.\n");
	printf("Verify Check [TestPattern1:%08x]\n", DDR_VERIFY_PATTERN1);
#endif
	if (ucDDR_verify(&ret, DDR_VERIFY_PATTERN1) > 0)
	{
		if(ret.result != RET_NG_RX)
		{
			output_error_info(&ret);
		}
		uiRTC_GetTimeSec();
#ifdef CONFIG_HW_WATCHDOG
		WATCHDOG_RESET();
#endif
		return ret.result;
	}
	
#if 0 /* original code */
	/* Verify Check 2 */
	usart_puts("Verify Check [TestPattern2]\n");
#else
	printf("Verify Check [TestPattern2:%08x]\n", DDR_VERIFY_PATTERN2);
#endif
	if (ucDDR_verify(&ret, DDR_VERIFY_PATTERN2) > 0)
	{
		if(ret.result != RET_NG_RX)
		{
			output_error_info(&ret);
		}
		return ret.result;
	}
	
#if 0 /* original code */
	usart_puts("Verify OK.\n");
#else
	printf("Verify OK.\n");
#endif
	
	return ret.result;
}


/*-----------------------------------------------------------------------------
 * DDR2 Own Address Write/Readd Check
 *    RETRUN value : (unsigned char) RET_OK, RET_NG, RET_NG_RX(RXRDY detect)
 *---------------------------------------------------------------------------*/
unsigned char ucDDR2_AddrOwn(void)
{
	unsigned int addr;
	unsigned int data;
	struct ddr_result ret = { RET_NG, AT91C_BASE_DDRCS, 0x00000000, 0x00000000};
	
#if ENABLE_INTRRUPT_COMANND /* Interrupt command check */
	/* Check Receiver Ready */
	if (ucUSART3_RXRDY_detect() == RXRDY_DETECT)
	{
		ret.result = RET_NG_RX;
		return ret.result;
	}
	/* Own Address Write */
	usart_puts("Addr Write... ");
#else
	printf("Addr Write... ");
#endif
	for (addr = 0; addr < DDR_SIZE; addr += 4)
	{
		write_ddr(addr, (AT91C_BASE_DDRCS + addr));		/* DDR2 Write */
	}
	uiRTC_GetTimeSec();
#ifdef CONFIG_HW_WATCHDOG
	WATCHDOG_RESET();
#endif

#if ENABLE_INTRRUPT_COMANND /* Interrupt command check */
	/* Check Receiver Ready */
	if (ucUSART3_RXRDY_detect() == RXRDY_DETECT)
	{
		ret.result = RET_NG_RX;
		return ret.result;
	}
	/* Read Check */
	usart_puts("\nRead check start\n");
#else
	printf("\nRead check start\n");
#endif
	for (addr = 0; addr < DDR_SIZE; addr += 4) {
		data = read_ddr(addr);							/* DDR2 Read */
		if (data != (AT91C_BASE_DDRCS + addr)) {
			ret.result       = RET_NG;
			ret.address      = AT91C_BASE_DDRCS + addr;
			ret.expect_value = ret.address;
			ret.actual_value = data;
			output_error_info(&ret);
			return ret.result;
		}
	}
	uiRTC_GetTimeSec();
#ifdef CONFIG_HW_WATCHDOG
	WATCHDOG_RESET();
#endif

#if 0 /* original code */
	usart_puts("Addressing test OK.\n");
#else
	printf("Addressing test OK.\n");
#endif
	
	return RET_OK;
}

int check_dipswitch(void)
{
  int ret = 0;
  unsigned int val;
  
  /* INPUT GPIO31(GPIO1[2]) */
  val =  readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_EXT_PORTA);
  if ( val & BIT(2) ){
    ret |= 0x04;
    /* printf("***** normal mode: ON\n"); */
  }
  /* INPUT GPIO41(GPIO1[12] */
  val =  readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_EXT_PORTA);
  if ( val & BIT(12) ){
    ret |= 0x02;
    /* printf("***** diag mode1: ON\n"); */
  }
  /* INPUT GPIO37(GPIO1[8]) */
  val =  readl(SOCFPGA_GPIO1_ADDR + SOCFPGA_GPIO_EXT_PORTA);
  if ( val & BIT(8) ){
    ret |= 0x01;
    /* printf("***** diag mode2: ON\n"); */
  }
  return ret;

}

/*
 * read DIP-SW for diag.  
 * Read 3 times at 10usec cycle and match 3 times.
*/
int check_dipswitch_proc(void)
{
  int buff[3];
  int i=0, ret=0;
  
  for(i = 0; i < 3; i++){
    buff[i] = check_dipswitch();
    
    if(i < 2)
      udelay(10);
    else
      break;
  }
  /* Update DIP-SW data*/
  ret |= buff[0] & buff[1] & buff[2]; /* High logic. */
  ret &= buff[0] | buff[1] | buff[2]; /* Low logic. */
  
  return ret;
}

void diag_check(void)
{
	int dsw = 0;

	dsw = check_dipswitch_proc();

	if (dsw == 2) {
#if 0 /* original code */
		usart_puts("mode 1\n");
#else
		printf("***** boot: diag mode 1\n");
#endif
#if 0           /* for debug */
		ucDDR2_ReadWrite();
		ucDDR2_AddrOwn();
		ucEth_Int_Check();
#endif
		diag_main();
		
	} else if (dsw == 1) {
#if 0 /* original code */
		usart_puts("mode 2\n");
#else
		printf("***** boot: diag mode 2\n");
#endif
	} else {
#if 0 /* original code */
		usart_puts("mode normal ");
#else
		printf("***** boot: normal mode\n");
#endif
	}
}

enum DIAG_PROC{
  DIAG_PROC_ALL=0,
  DIAG_PROC_DDR_DBC,
  DIAG_PROC_DDR_ABC,
  DIAG_PROC_ETH_IC,
  DIAG_PROC_MAX,
};

/*
 * Diag Check function for u-boot console Command.
 */
int do_diagchk(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
  int proc=DIAG_PROC_MAX, ret=0;
  char *end;
  
  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  switch (argc) {
  case 2:
    proc = (int)simple_strtoul(argv[1], &end, 10);
    break;

  case 1:
    break;
    
  default:
    printf("Invalid parameter!\n");
    return CMD_RET_USAGE;
  }

  /* printf("***** %s:%s:%d\n", __FILE__, __FUNCTION__, __LINE__); */
  switch (proc) {
  case DIAG_PROC_ALL:
    ucDDR2_ReadWrite();
    ucDDR2_AddrOwn();
    ucEth_Int_Check();
    break;
    
  case DIAG_PROC_DDR_DBC:
    ucDDR2_ReadWrite();
    break;
    
  case DIAG_PROC_DDR_ABC:
    ucDDR2_AddrOwn();
    break;

  case DIAG_PROC_ETH_IC:
    ucEth_Int_Check();
    break;
    
  case DIAG_PROC_MAX:
#ifndef CONFIG_DIAG_DEBUG
    diag_check();
#else
    diag_main();
#endif
    break;
    
  default:
    printf("Invalid parameter!\n");
    return CMD_RET_USAGE;
  }
  return ret;
}

/* //////////////////////////////////////////////////////////////////////////////////////////// */
/* Porting From PT 6                                                                            */
/* //////////////////////////////////////////////////////////////////////////////////////////// */
/*-----------------------------------------------------------------------------
 *  Software Reset
 *    RETRUN value : none
 *---------------------------------------------------------------------------*/
static void vBoardReset(void)
{
	reset_cpu(0);
}


/* for Debug */
static void usart_puts(const char *ptr)
{
	puts(ptr);
}


static  char usart_getc(void)
{
	return getc();
}


/* need to port functions -------------------------------------------------------- */
/* for Diag Protocol */
static void vUSART3_hw_init(void)
{
	volatile unsigned long reg;
	/* baudrate 75Kにかえて、パリティを偶数にしなければならない */
	if(USARTn == UART0){
	        printf("%s ---\n", USART_FOR_DIAG);
		if(serial_assign(USART_FOR_DIAG) != 0){
	    		printf("not found %s\n", USART_FOR_DIAG);
		}
  	} 
  	else if(USARTn == UART1){
		printf("%s---\n", USART_FOR_DIAG);
		/* console : serial to eserial0, current serial: eserial1 */
		console_assign(stdin , USART_FOR_LOG);
		console_assign(stdout, USART_FOR_LOG);
		console_assign(stderr, USART_FOR_LOG);

	        if(serial_assign(USART_FOR_DIAG) != 0){
		    printf("not found %s\n", USART_FOR_DIAG);
		}
		serial_init();
	}
	else{
		return;
	}
#ifndef CONFIG_DIAG_DEBUG
	/* direct control */
	reg = read_usart3(USARTn + USARTn_SFE);
	reg &= 0xFFFFFFFE;         /* disable */
	write_usart3(USARTn + USARTn_SFE, reg);

	reg = read_usart3(USARTn + USARTn_FMT);
	reg &= 0xFFFFFFE7;         /* around-parity setting clear(to hard-default) */
	reg |= 0x00000018;         /* parity enable & parity_even */
	write_usart3(USARTn + USARTn_FMT, reg);

	reg = read_usart3(USARTn + USARTn_FIFO);
	reg |= 0x00000006;         /* fifo reset */
	write_usart3(USARTn + USARTn_FIFO, reg);

	/* baudrate */
	reg = read_usart3(USARTn + USARTn_FMT);
	reg |= 0x80UL;
	write_usart3(USARTn + USARTn_FMT, reg); /* baudrate reg Access enable */

	/* baudrate 75K */
	reg = 0x00000053;
	write_usart3(USARTn + USARTn_DLL, reg);
	reg = 0x00000000;
	write_usart3(USARTn + USARTn_DLH, reg);

	reg = read_usart3(USARTn + USARTn_FMT);
	reg &= 0xFFFFFF7F;
	write_usart3(USARTn + USARTn_FMT, reg); /* baudrate reg Access enable */

	for(reg = 0; reg < 8 * 6 + 20; ++reg); /* 8 l4 clk + margin */
  
	reg = read_usart3(USARTn + USARTn_SFE);
  	reg |= 0x00000001;         /* enable */
  	write_usart3(USARTn + USARTn_SFE, reg);
#else
#endif
}

/*-----------------------------------------------------------------------------
 * USART3 RXRDY detect
 *    RETRUN value : (unsinged char) 0(RXRDY_NON) = not detect,
 *                                   1(RXRDY_DETECT) = detect
 *---------------------------------------------------------------------------*/
static unsigned char ucUSART3_RXRDY_detect(void)
{
        volatile int reg;

	reg = serial_tstc();
	if(reg != 0){
		reg = 1;
	}
#ifndef CONFIG_DIAG_DEBUG
#else
	if(reg == 0x01){
		usart_puts("[hit]");
	}
#endif
	return reg;
}


/*-----------------------------------------------------------------------------
 * USART3 Get Character(1Byte)
 *    RETRUN value : (unsinged char) 1Byte data
 *---------------------------------------------------------------------------*/
static unsigned char ucUSART3_get_char(unsigned char *data)
{
        unsigned char ret;
	unsigned int value = read_usart3(USARTn + USARTn_LSR) &
				(AT91C_USART_PARE | AT91C_USART_FRAME | AT91C_USART_OVRE);
	
	if (value != 0)
	{
		if((value & AT91C_USART_PARE) != 0)
		{
			usart_puts("Parity Error\n");
		}
		if((value & AT91C_USART_FRAME) != 0)
		{
			usart_puts("Framing Error\n");
		}
		if((value & AT91C_USART_OVRE) != 0)
		{
			usart_puts("Overrun Error\n");
		}
#if 0 /* already clear when read reg. */
		write_usart3(USART_CR, AT91C_USART_RSTSTA);
#endif		
		ret = 1;
	}
	else
	{
		ret = 0;
	}
	/* Get Receive data & RXRDY flag clear */
        *data = serial_getc();
	
	return ret;
}

static void usart3_putc(const char c)
{
        serial_putc(c);
}

/*-----------------------------------------------------------------------------
 * USART3 Transmit Telegram
 *    RETRUN value : none
 *    unsigned char *buf : Telegram
 *---------------------------------------------------------------------------*/
static void vUSART3_send_telegram(unsigned char *buf)
{
	unsigned char length = buf[1] + 0x3;	/* LEN + {STX(1BYTE) + LEN(1Byte) + BCC(1Byte)} */
        unsigned int index;
	for(index = 0; index < length; index++)
	{
		usart3_putc((const char)buf[index]);
	}
}

/* for short period counter */
/*-----------------------------------------------------------------------------
 * Wait timer for State check (used: TC1=TC0.1ch)
 *    RETRUN value : none
 *---------------------------------------------------------------------------*/
static void vTC1_init(void)
{
        volatile unsigned int reg;
	/* TC1 ------------------------ */
	reg   = read_tc(TC1 + TC_CTRL);
	reg  &= 0xFFFFFFFEUL;
        write_tc(TC1 + TC_CTRL, reg);  /* stop */

	reg   = read_tc(TC1 + TC_EOI);     /* int clear */

	reg   = 0xFFFFFFFFUL;
	write_tc(TC1 + TC_LDCNT, reg); /* load value set. */

	reg   = read_tc(TC1 + TC_CTRL);
	reg  |= 0x00000006UL;              /* Evnet timer & interrupt mask; */
	write_tc(TC1 + TC_CTRL, reg);

	reg   = read_tc(TC1 + TC_EOI);
	/* RTC ------------------------ */
	reg   = read_tc(RTC + TC_CTRL);
	reg  &= 0xFFFFFFFEUL;
        write_tc(RTC + TC_CTRL, reg);  /* stop */

	reg   = read_tc(RTC + TC_EOI);     /* int clear */

	reg   = RTC_30sec;
	write_tc(RTC + TC_LDCNT, reg); /* load value set. */

	reg   = read_tc(RTC + TC_CTRL);
	reg  |= 0x00000006UL;              /* Evnet timer & interrupt mask; */
	write_tc(RTC + TC_CTRL, reg);

	reg   = read_tc(RTC + TC_EOI);

	reg   = read_tc(RTC + TC_CTRL);
	reg  |= 0x00000001UL;              /* Evnet timer & interrupt mask;Enable */
	write_tc(RTC + TC_CTRL, reg);
}

/*-----------------------------------------------------------------------------
 * Timer start
 *    RETRUN value : none
 *    unsigned int channel : Timer Channel (TC0 or TC1)
 *---------------------------------------------------------------------------*/
static void vTC_start(unsigned int channel)
{
        volatile unsigned long reg;
	reg   = 0xFFFFFFFFUL;
	write_tc(channel + TC_LDCNT, reg); /* load value set. */

	reg = read_tc(channel + TC_EOI);
        reg = read_tc(channel + TC_CTRL);
	reg |= 0x01UL;
  	write_tc(channel + TC_CTRL, reg); /* Enable */
}


/*-----------------------------------------------------------------------------
 * Timer stop
 *    RETRUN value : none
 *    unsigned int channel : Timer Channel (TC0 or TC1)
 *---------------------------------------------------------------------------*/
static void vTC_stop(unsigned int channel)
{
        volatile unsigned long reg; 

	reg   = read_tc(channel + TC_CTRL);
	reg  &= 0xFFFFFFFEUL;
        write_tc(channel + TC_CTRL, reg);  /* stop */
}


/*-----------------------------------------------------------------------------
 * TImer reset
 *    RETRUN value : none
 *    unsigned int channel : Timer Channel (TC0 or TC1)
 *---------------------------------------------------------------------------*/
static void vTC_reset(unsigned int channel)
{
        volatile unsigned long reg; 

	reg   = read_tc(channel + TC_CTRL);
	reg  &= 0xFFFFFFFEUL;
        write_tc(channel + TC_CTRL, reg);  /* stop */

	reg   = read_tc(channel + TC_EOI);     /* int clear */

	reg   = 0xFFFFFFFFUL;
	write_tc(channel + TC_LDCNT, reg); /* load value set. */

	reg   = read_tc(channel + TC_EOI);

	reg   = read_tc(channel + TC_CTRL);
	reg  |= 0x00000006UL;              /* Evnet timer & interrupt mask; */
	write_tc(channel + TC_CTRL, reg);

	vTC_start(channel);
}


/*-----------------------------------------------------------------------------
 * Get Counter value
 *    RETRUN value : (unsinged int) counter value
 *    unsigned int channel : Timer Channel (TC0 or TC1)
 *---------------------------------------------------------------------------*/
static unsigned int uiTC_get_value(unsigned int channel)
{
        volatile  unsigned long reg;

  	reg   = read_tc(channel + TC_CTRL);
	reg  &= 0x1UL;

	if(reg != 0){
		reg = read_tc(channel + TC_RAWINT);
	  	reg &= 0x01UL;
		if(reg == 1){
			/* not clear interrupt signal */
			return 0xFFFFFFFFUL;
		}
		else{
			return 0xFFFFFFFFUL - read_tc(channel + TC_CURCNT);
	  	}
	}
	else{
		return 0;
	}
}

/*-----------------------------------------------------------------------------
 *  RTC register value  -> second value (AM: +0 hour, PM: +12 hour)
 *    RETRUN value : RTC value (Unit: sec)
 *---------------------------------------------------------------------------*/
static unsigned int uiRTC_GetTimeSec(void)
{
        volatile unsigned long reg;
  
        reg = read_tc(RTC + TC_CTRL);
	reg &= 0x01UL;

	if(reg == 0){
		rtc_time = 0;
		return 0;
	}

	reg = read_tc(RTC + TC_RAWINT);
	reg &= 0x01UL;
	if(reg == 1){
		rtc_time += 30;
        	reg = read_tc(RTC + TC_EOI);
#ifndef CONFIG_DIAG_DEBUG
#else
        	printf("[time] %d sec.\n", rtc_time);
#endif	  
	}

	return rtc_time;
}
/* end of need to port functions ------------------------------------------- */

 
/*-----------------------------------------------------------------------------
 * Self Check Mode 1 Timeout checker
 *    RETRUN value : none
 
 *---------------------------------------------------------------------------*/
static void vSelfCheck_Mode_time(unsigned int limit)
{
	unsigned int current_time;
	
	current_time = uiRTC_GetTimeSec();
		
	if(current_time >= limit)
	{
		usart_puts("=== Self Check Mode 1 : TIME OUT ===\n");
		vBoardReset();
	}
}


/*-----------------------------------------------------------------------------
 *  Self Check Mode 1 Timeout value set
 *    RETRUN value : Limit time (Unit: sec)
 *    unsigned int limit_min : Configurtion time(Unit: min)
 *---------------------------------------------------------------------------*/
static unsigned int uiSelfCheck_limit_set(unsigned int limit_min)
{
	unsigned int lim_time;
	
	lim_time = limit_min * 60;
	
	return lim_time;
}


/*-----------------------------------------------------------------------------
 * Self Check mode 1, Function select & Execute
 *    RETRUN value : (unsinged char) RET_OK,
 *                                   RET_NG,
 *                                   RETURN_VALUE_NONE,
 *                                   RET_NG_RX, (RXRDY detect)
 *                                   RET_NG_NOT_FOUND (EtherPHY only)
 *---------------------------------------------------------------------------*/
static unsigned char ucDIAG_selec_function(unsigned char proc_number)
{
	unsigned char ret = 0;
	switch (proc_number) {
	        case 0x03:	ret = ucDDR2_ReadWrite();	                break;
	        case 0x04:	ret = ucDDR2_AddrOwn();                         break;
		case 0x05:	ret = ucEth_Int_Check();	                break;
		case 0x1E:
		case 0x1F:	ret = ucLPWA_Output(proc_number);               break;
		case 0x20:
		case 0x21:
		case 0x22:
		case 0x23:	ret = ucLPWA_Input(proc_number);                break;
		default:
			ret = 1;
			usart_puts("- invalid_input.\n\n");
			break;
	}
	return ret;
}

/*-----------------------------------------------------------------------------
 * Telegram Generate ( NW Board -> Machine )
 *    RETRUN value : none
 *    unsigned char *buf        : telegram
 *    unsigned char test_type   : test type
 *    unsigned char proc_number : process number
 *    unsigned char result      : self check result("OK" or "NG" or "return value none" )
 *---------------------------------------------------------------------------*/
static void vDIAG_telegram_gen(unsigned char *buf, unsigned char test_type, unsigned char proc_number, unsigned char result)
{
	unsigned char n=0;
	/* STX */
	buf[n++] = MSG_STX;
	
	/* Length */
	buf[n++] = (result == RETURN_VALUE_NONE) 
				? 2		/* TYPE & ProcNo */
				: 3;	/* TYPE & ProcNo & Result */
	
	/* BOAD_TYPE & CHECK_TYPE */
	buf[n++] = (MSG_BOAD_TYPE_NW << 4) | test_type;
	
	/* Process Number */
	buf[n++] = proc_number;
	
	/* Result value */
	if(result != RETURN_VALUE_NONE)
	{
		buf[n++] = result;
	}
	
	/* BCC */
	buf[n] = ucDIAG_calculateBCC(buf, n);
}


/*-----------------------------------------------------------------------------
 * Telegram Error Check
 *    RETRUN value : (unsigned char) 0 = OK, 1 = NG
 *    unsigned char *buf        : telegram
 *    unsigned char test_type   : test type
 *    unsigned char proc_number : process number
 *---------------------------------------------------------------------------*/
static unsigned char ucDIAG_telegram_check(unsigned char *buf, unsigned char *test_type, unsigned char *proc_number)
{
	/* Check Field: STX */
	if(buf[0] != MSG_STX)
	{
		usart_puts("Invalid STX(0x");
		output_hex_8bit(buf[0]);
		usart_puts(")\n");
		return 1;
	}
	
	/* Get Field: Length */
	unsigned char data_length = buf[1];
	if(data_length > DATA_LENGTH_MAX)
	{
		usart_puts("Invalid Data Length(0x");
		output_hex_8bit(data_length);
		usart_puts(")\n");
		return 1;
	}
	
	/* Check Board Type */
	unsigned char board_type = ((buf[2] & 0xF0) >> 4);
	if(board_type != MSG_BOAD_TYPE_NW)
	{
		usart_puts("Invalid board type(0x");
		output_hex_8bit(board_type);
		usart_puts(")\n");
		return 1;
	}
	
	/* Check Test type */
	unsigned char tmp_test_type = buf[2] & 0x0F;
	if((tmp_test_type != MSG_CHECK_TYPE_INPUT) && (tmp_test_type != MSG_CHECK_TYPE_OUTPUT) && (tmp_test_type != MSG_CHECK_TYPE_OTHER))
	{
		usart_puts("Invalid check type(0x");
		output_hex_8bit(tmp_test_type);
		usart_puts(")\n");
		return 1;
	}
	*test_type = tmp_test_type;
	
	/* Check Test Process Nomber */
	unsigned char tmp_proc_no = buf[3];
	if((tmp_proc_no < PROC_NO_MIN) || (tmp_proc_no > PROC_NO_MAX))
	{
		usart_puts("Invalid procNo(0x");
		output_hex_8bit(tmp_proc_no);
		usart_puts(")\n");
		return 1;
	}
	*proc_number = buf[3];
	
	/* Check BCC */
	unsigned char bcc = ucDIAG_calculateBCC(buf, (data_length + 2));
	if(bcc != buf[data_length + 2]){
		usart_puts("Invalid BCC(0x");
		output_hex_8bit(buf[data_length + 2]);
		usart_puts(")\n");
		return 1;
	}
	return 0;
}


/*-----------------------------------------------------------------------------
 * Calculate BCC
 *    RETRUN value : (unsigned char) bcc
 *    unsigned int *frame       : target data
 *    unsigned char data_length : target data length
 *---------------------------------------------------------------------------*/
static unsigned char ucDIAG_calculateBCC(unsigned char	*frame, unsigned char data_length)
{
	unsigned char n = 0;
	unsigned char bcc = 0;

	bcc = *frame;
	for (n = 1; n < data_length; n++) {
		bcc ^= (*(frame + n));
	}

	return bcc;
}


static void output_hex8(unsigned char byte)
{
	switch (byte) {
		case 0 :usart_puts("0");		break;
		case 1 :usart_puts("1");		break;
		case 2 :usart_puts("2");		break;
		case 3 :usart_puts("3");		break;
		case 4 :usart_puts("4");		break;
		case 5 :usart_puts("5");		break;
		case 6 :usart_puts("6");		break;
		case 7 :usart_puts("7");		break;
		case 8 :usart_puts("8");		break;
		case 9 :usart_puts("9");		break;
		case 10:usart_puts("A");		break;
		case 11:usart_puts("B");		break;
		case 12:usart_puts("C");		break;
		case 13:usart_puts("D");		break;
		case 14:usart_puts("E");		break;
		case 15:usart_puts("F");		break;
		default:				break;
	}
}

/*
 * 8bit hex output
*/
static void output_hex_8bit(unsigned char byte)
{
	output_hex8((0xF0 & byte) >> 4);
	output_hex8(0x0F & byte);
}


/*
 *  32bit hex output (0x add.)
 */
static void output_hex(unsigned int num)
{
	unsigned char i;
	unsigned char byte;
	usart_puts("0x");

	i = 32;
	do{
		i-=8;
		byte = (unsigned char)(0x000000FF & (num >> i));

		output_hex8((unsigned char)(byte/16)); /* output_hex8(div(byte,16)); */
                output_hex8((unsigned char)(byte%16)); /* output_hex8(mod(byte,16)); */

	}while(i>0);
}

/*-----------------------------------------------------------------------------
 * Self Check Mode 1 : Main routine
 *    RETRUN value : none
 *---------------------------------------------------------------------------*/
static void diag_main(void)
{
	unsigned int limit_time;
	unsigned char buf[8];
	unsigned char buf_size = (sizeof(buf) / sizeof(buf[0]));
	unsigned char i;
	unsigned char ret;
	unsigned char test_type;
	unsigned char proc_number;
	
	/* UART use setting */
#ifndef CONFIG_DIAG_DEBUG
#else
	printf("------ Information for ddr Diag check(debug) ------\n");
	printf("SDRAM start   Address : 0x%08x\n", CONFIG_SYS_SDRAM_BASE);
	printf("UBOOT start   Address : 0x%08x\n", gd->relocaddr);
	printf("diag func     Address : 0x%08x\n", (unsigned long)diag_check);
	printf("DDR chk start Address : 0x%08x\n", AT91C_BASE_DDRCS);
	printf("DDR chk end   Address : 0x%08x\n", DDR_SIZE + AT91C_BASE_DDRCS);
	printf("GD            Address : 0x%08x\n", (unsigned long)gd);
	printf("SP TOP        Address : 0x%08x\n", gd->start_addr_sp);
	printf("------ Information for ddr Diag check(debug) ------\n");
#endif	
	vUSART3_hw_init();
	
	/* SelfCheck Mode 1 limit Time Set */
	limit_time = uiSelfCheck_limit_set(MODE1_LIMIT_TIME);
	
	/* Timer Initialize (For UART com.)*/
	vTC1_init();

	/* Reseive Buffer clear */
/*	ucUSART3_get_char(&buf[0]); */ /* should be done at initialize. and Reading at this timing depends on the other model. */
	
	i = 0;
	while(1){
		/* Self check mode 1 : Timeout check */
		vSelfCheck_Mode_time(limit_time);
#ifdef CONFIG_HW_WATCHDOG
		WATCHDOG_RESET();
#endif
	
		if (ucUSART3_RXRDY_detect() == RXRDY_DETECT)
		/* Complete character has been received */
		{
			/* TImer reset & stop*/
			vTC_stop(TC1);
			
			if(ucUSART3_get_char(&buf[i]) == 0)
			/* No Error */
			{
				i++;
				/* Buffer oVerflow check */
				if (i >= buf_size)
				{
					memset(buf, 0x0, buf_size);
					usart_puts("Buffer overflow.\n");
					i = 0;
				}
				else
				{
					/* Confirmed check timer start */
					vTC_start(TC1);
				}
			}
			else
			/* Parity or Framing or Overrun Error */
			{
				i = 0;
			}
		}
		else
		/* No complete character has been received. */
		{
			if ( uiTC_get_value(TC1) > TIME_10ms )
			/* 10ms passage */
			{
				/* Receive Telegram Error check */
				if (ucDIAG_telegram_check(buf, &test_type, &proc_number) > 0){
					i = 0;
					memset(buf, 0x0, buf_size);
					usart_puts("Telegram Error\n");
				}
				else
				{
					/* Execute self check function */
					ret = ucDIAG_selec_function(proc_number);
					i = 0;
					if (ret >= RET_ABORT)
					{
						/* RXRDY Detect */
						memset(buf, 0x0, buf_size);
						proc_number = 0;
					}
					else{
						/* Send Telegram Generation */
						memset(buf, 0x0, buf_size);
						vDIAG_telegram_gen(buf, test_type, proc_number, ret);
						
						while( uiTC_get_value(TC1) < TIME_20ms )
						{
							if (ucUSART3_RXRDY_detect() == RXRDY_DETECT)
							/* Complete character has been received */
							{
								memset(buf, 0x0, buf_size);
								break;
							}
						}
						vUSART3_send_telegram(buf);
						usart_puts("Process End.\n");
					}
				}
				vTC_stop(TC1);
			}
		}
	}
}

/* //////////////////////////////////////////////////////////////////////////////////////////// */
/* Porting From PT 6                                                                            */
/* //////////////////////////////////////////////////////////////////////////////////////////// */

U_BOOT_CMD(
	diagchk,	2,	1,	do_diagchk,
	"EF-NWSoC: Start diagnosis process",
	"[siagnosis Number]\n"
	" 0 : Check All process.\n"
	" 1 : Check DRAM DATA BUS.\n"
	" 2 : Check DRAM ADRRESS BUS.\n"
	" 3 : Check ETHER PHY Intrrupt.\n"
	" 4 or NULL : Check All process with DIP-SW\n"
);

/* zimage (little-endian) */
#define	LINUX_ZIMAGE_MAGIC	0x016f2818

typedef struct linux_zimage_header {
	unsigned int	code[9];
	unsigned int	magic;
	unsigned int	start;
	unsigned int	end;
}zimage_hdr_t;

/* fdt (big-endian) */
#define FDT_MAGIC		0xd00dfeed

typedef struct fdt_header {
  u_int magic;
  u_int size;
  u_int off_dt_struct;
  u_int off_dt_strings;
  u_int off_mem_rsvmap;
  u_int version;
  u_int last_comp_version;
  u_int boot_cpuid_phys;
  u_int size_dt_strings;
  u_int size_dt_struct;
}fdt_hdr_t;

/*
 * set imagesize for Linux and fdt.
 */
static int do_setenv_imagesize(void)
{
  int ret = 0;
  nand_info_t *nand = &nand_info[0];
  char *ep;
  size_t rwsize, isize;
  zimage_hdr_t hdr_kernel;
  fdt_hdr_t hdr_fdt;
  loff_t off;
  char istr[11];

  /* KERNEL */
  ep = getenv("nandbootimageaddr");
  if (ep != NULL)
    off = (ulong)simple_strtoul(ep, NULL, 16);
  else
    goto error_proc_kernel;

  rwsize = sizeof(zimage_hdr_t);
  ret = nand_read_skip_bad(nand, off, &rwsize, (u_char *)&hdr_kernel);

  if((ret == 0) && cpu_to_le32(hdr_kernel.magic) == LINUX_ZIMAGE_MAGIC){
    isize = cpu_to_le32(hdr_kernel.end) - cpu_to_le32(hdr_kernel.start);
    sprintf(istr, "0x%08x", (ulong)isize);
    setenv("bootimagesize", istr);
  }else{
    printf("ERROR: Invalid magic number!!\n");
  error_proc_kernel:
    setenv("bootimagesize", __STR(CONFIG_SYS_LOAD_SIZE));
    printf("Set default bootimagesize.\n");
  }    

  /* FDT */
  ep = getenv("nandfdtaddr");
  if (ep != NULL)
    off = (ulong)simple_strtoul(ep, NULL, 16);
  else
    goto error_proc_fdt;

  rwsize = sizeof(fdt_hdr_t);
  ret = nand_read_skip_bad(nand, off, &rwsize, (u_char *)&hdr_fdt);

  if((ret == 0) && cpu_to_be32(hdr_fdt.magic) == FDT_MAGIC){
    isize = cpu_to_be32(hdr_fdt.size);
    sprintf(istr, "0x%08x", (ulong)isize);
    setenv("fdtimagesize", istr);
  }else{
    printf("ERROR: Invalid magic number!!\n");
  error_proc_fdt:
    setenv("fdtimagesize", __STR(CONFIG_SYS_FDT_SIZE));
    printf("Set default fdtimagesize.\n");
  }
  return 0;

}

U_BOOT_CMD(
	setenv_imagesize,	1,	1,	do_setenv_imagesize,
	"EF-NWSoC: Set envitonment bootimagesize from zImage header descritor.",
	" No argument.\n"
);
  
